<?php

namespace App\Console\Commands;

use App\Models\EventMedia;
use Illuminate\Console\Command;

/**
 * FIX #16: Cleanup soft-deleted media after retention period
 */
class CleanupDeletedMedia extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'media:cleanup-deleted {--days=30 : Days after which to permanently delete} {--force : Skip confirmation prompt}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Permanently delete media that has been soft-deleted for specified days';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $days = (int) $this->option('days');
        $date = now()->subDays($days);

        $this->info("Searching for media deleted more than {$days} days ago...");

        $count = EventMedia::onlyTrashed()
            ->where('deleted_at', '<', $date)
            ->count();

        if ($count === 0) {
            $this->info('No media to cleanup.');
            return 0;
        }

        $this->warn("Found {$count} media files ready for permanent deletion.");
        $this->comment("These files were deleted before: {$date->format('Y-m-d H:i:s')}");

        if (!$this->option('force') && !$this->confirm('Permanently delete these files? This cannot be undone.', false)) {
            $this->info('Cleanup cancelled.');
            return 0;
        }

        $this->output->progressStart($count);

        EventMedia::onlyTrashed()
            ->where('deleted_at', '<', $date)
            ->chunk(100, function ($mediaItems) {
                foreach ($mediaItems as $media) {
                    $media->forceDelete();
                    $this->output->progressAdvance();
                }
            });

        $this->output->progressFinish();

        $this->info("\n✓ Successfully deleted {$count} media files permanently.");
        $this->comment('  Physical files and database records have been removed.');

        return 0;
    }
}
