<?php

namespace App\Console\Commands;

use App\Models\Event;
use Illuminate\Console\Command;

class CleanupExpiredFaceData extends Command
{
    /**
     * FIX #22: GDPR compliance - cleanup face recognition data for expired events
     *
     * @var string
     */
    protected $signature = 'gdpr:cleanup-face-data';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Delete face recognition data for expired events (GDPR compliance)';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Scanning for expired events with face data...');

        $expiredEvents = Event::whereNotNull('expiry_date')
            ->where('expiry_date', '<', now())
            ->with('media.faces')
            ->get();

        if ($expiredEvents->isEmpty()) {
            $this->info('No expired events found.');
            return 0;
        }

        $totalFaces = 0;
        $totalEvents = 0;

        foreach ($expiredEvents as $event) {
            $eventFaceCount = 0;

            foreach ($event->media as $media) {
                $faceCount = $media->faces()->count();
                if ($faceCount > 0) {
                    $media->faces()->delete();
                    $eventFaceCount += $faceCount;
                }
            }

            if ($eventFaceCount > 0) {
                $totalFaces += $eventFaceCount;
                $totalEvents++;
                $this->line("  - Event #{$event->id} ({$event->name}): Deleted {$eventFaceCount} face embeddings");
            }
        }

        if ($totalFaces > 0) {
            $this->info("\n✓ Cleaned up {$totalFaces} face embeddings from {$totalEvents} expired events.");
            $this->comment('  (GDPR compliance: Biometric data deleted after event expiry)');
        } else {
            $this->info('No face data found in expired events.');
        }

        return 0;
    }
}
