<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Symfony\Component\Process\Process;

class DevCommand extends Command
{
    protected $signature = 'serve {--host=127.0.0.1} {--port=8000} {--workers=4 : Number of queue workers}';
    protected $description = 'Start dev server + queue workers + scheduler together';

    public function handle(): void
    {
        $php  = PHP_BINARY;
        $base = base_path();
        $host = $this->option('host');
        $port = $this->option('port');

        $workerCount = max(1, (int) $this->option('workers'));

        $this->info("Starting development environment...");
        $this->line("  <fg=green>Server</>    → http://{$host}:{$port}");
        $this->line("  <fg=yellow>Queue</>     → {$workerCount} workers (default + AI) running in background");
        $this->line("  <fg=cyan>Scheduler</> → running in background");
        $this->line("  Press <fg=red>Ctrl+C</> to stop all.");
        $this->newLine();

        // Use PHP built-in server directly (run from public/ dir like Laravel's ServeCommand does)
        $serverScript = base_path('vendor/laravel/framework/src/Illuminate/Foundation/resources/server.php');
        $server = new Process([$php, '-S', "{$host}:{$port}", $serverScript], public_path(), [
            'APP_ENV'       => config('app.env', 'local'),
            'APP_BASE_PATH' => $base,
        ]);

        // Start multiple queue workers for parallel image processing
        $workers = [];
        for ($i = 0; $i < $workerCount; $i++) {
            $worker = new Process([$php, 'artisan', 'queue:work', '--queue=default', '--tries=3', '--timeout=600', '--sleep=1'], $base);
            $worker->setTimeout(null);
            $worker->start();
            $workers[] = $worker;
        }

        $scheduler = new Process([$php, 'artisan', 'schedule:work'], $base);

        $server->setTimeout(null);
        $scheduler->setTimeout(null);

        $scheduler->start();

        // Server output goes directly to terminal
        $server->start(function (string $type, string $output) {
            $this->output->write($output);
        });

        $server->wait();

        foreach ($workers as $worker) {
            $worker->stop();
        }
        $scheduler->stop();

        $this->newLine();
        $this->info("All processes stopped.");
    }
}
