<?php

namespace App\Console\Commands;

use App\Models\EventMedia;
use App\Services\CloudStorageManager;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;

class MigrateMediaToPrivate extends Command
{
    protected $signature = 'media:migrate-to-private {--dry-run : Show what would be moved without moving}';

    protected $description = 'Migrate event media files from public storage to private storage';

    public function handle(CloudStorageManager $cloudStorage): int
    {
        $cloudStorage->registerDisks();

        $dryRun = $this->option('dry-run');

        if ($dryRun) {
            $this->info('DRY RUN — no files will be moved.');
        }

        $publicDisk = Storage::disk('public');
        $privateDisk = Storage::disk('client_media');

        $query = EventMedia::query()->where('disk', 'public');
        $total = $query->count();

        if ($total === 0) {
            $this->info('No media files on the public disk. Nothing to migrate.');

            return self::SUCCESS;
        }

        $this->info("Found {$total} media records on the public disk.");

        if (! $dryRun && ! $this->confirm("Migrate {$total} media files to private storage?")) {
            return self::SUCCESS;
        }

        $migrated = 0;
        $skipped = 0;
        $failed = 0;

        $bar = $this->output->createProgressBar($total);
        $bar->start();

        $query->chunkById(100, function ($mediaItems) use ($publicDisk, $privateDisk, $dryRun, &$migrated, &$skipped, &$failed, $bar) {
            foreach ($mediaItems as $media) {
                $paths = array_filter([
                    'original_path' => $media->original_path,
                    'optimized_path' => $media->optimized_path,
                    'thumbnail_path' => $media->thumbnail_path,
                ]);

                if (empty($paths)) {
                    $skipped++;
                    $bar->advance();

                    continue;
                }

                if ($dryRun) {
                    $migrated++;
                    $bar->advance();

                    continue;
                }

                try {
                    foreach ($paths as $path) {
                        if (! $publicDisk->exists($path)) {
                            continue;
                        }

                        // Copy file to private storage
                        $stream = $publicDisk->readStream($path);
                        if (! $stream) {
                            continue;
                        }

                        $privateDisk->putStream($path, $stream);
                        if (is_resource($stream)) {
                            fclose($stream);
                        }
                    }

                    // Update the database record to point to private disk
                    $media->update(['disk' => 'client_media']);

                    // Delete from public disk after successful migration
                    foreach ($paths as $path) {
                        $publicDisk->delete($path);
                    }

                    $migrated++;
                } catch (\Throwable $e) {
                    $failed++;
                    $this->newLine();
                    $this->error("Failed to migrate media #{$media->id}: {$e->getMessage()}");
                }

                $bar->advance();
            }
        });

        $bar->finish();
        $this->newLine(2);

        $this->info("Migration complete:");
        $this->line("  Migrated: {$migrated}");
        $this->line("  Skipped:  {$skipped}");
        $this->line("  Failed:   {$failed}");

        if ($dryRun) {
            $this->warn('This was a dry run. Run without --dry-run to actually migrate files.');
        }

        return $failed > 0 ? self::FAILURE : self::SUCCESS;
    }
}
