<?php

namespace App\Console\Commands;

use App\Services\GoogleDriveImportService;
use Illuminate\Console\Command;

class RunGoogleDriveImports extends Command
{
    protected $signature = 'imports:run-google-drive {--seconds=40} {--batch=15}';

    protected $description = 'Process pending Google Drive imports in small batches (cron-friendly).';

    public function handle(GoogleDriveImportService $service): int
    {
        $maxSeconds = (int) $this->option('seconds');
        $batchSize = (int) $this->option('batch');
        $started = microtime(true);
        $processedAny = false;

        while ((microtime(true) - $started) < $maxSeconds) {
            $import = $service->runBatch(null, $batchSize, $maxSeconds - (microtime(true) - $started));
            if (! $import) {
                break;
            }

            $processedAny = true;

            $this->info(
                sprintf(
                    'Import #%d status=%s processed=%d/%d failed=%d',
                    $import->id,
                    $import->status,
                    $import->processed_files,
                    $import->total_files,
                    $import->failed_files
                )
            );

            if ($import->status === 'completed' || $import->status === 'completed_with_errors') {
                continue;
            }
        }

        if (! $processedAny) {
            $this->info('No pending Google Drive imports.');
        }

        return Command::SUCCESS;
    }
}
