<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportActivityLogsRequest;
use App\Http\Requests\Admin\StoreActivityLogRequest;
use App\Http\Requests\Admin\UpdateActivityLogRequest;
use App\Models\ActivityLog;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Str;

class ActivityLogController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $action = $request->query('action');
        $userId = $request->query('user');
        $sort = $request->query('sort', 'latest');

        $query = ActivityLog::query()->with('user');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('description', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('subject_type', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('action', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($action) {
            $query->where('action', $action);
        }

        if ($userId) {
            $query->where('user_id', $userId);
        }

        $this->applySort($query, $sort);

        return view('admin.activity-logs.index', [
            'logs' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'action' => $action,
            'userId' => $userId,
            'sort' => $sort,
            'actions' => config('system.activity_types', []),
            'users' => User::query()->orderBy('name')->limit(200)->get(),
        ]);
    }

    public function notifications(Request $request)
    {
        $after = (int) $request->query('after', 0);
        $limit = (int) $request->query('limit', 6);
        $limit = max(1, min($limit, 20));

        $query = ActivityLog::query()->with('user')->orderByDesc('id')->orderByDesc('logged_at');

        $canViewAll = Gate::allows('viewAny', ActivityLog::class);
        if (! $canViewAll) {
            $query->where('user_id', $request->user()?->id);
        }

        if ($after > 0) {
            $query->where('id', '>', $after);
        }

        $logs = $query->limit($limit)->get();
        $latestId = $logs->max('id') ?? $after;

        $items = $logs->map(function (ActivityLog $log) {
            $subject = $log->subject_type ? class_basename($log->subject_type) : null;
            $summary = $log->description ?: trim(implode(' ', array_filter([$log->action, $subject])));
            $summary = $summary !== '' ? ucfirst($summary) : 'Activity update';

            return [
                'id' => $log->id,
                'summary' => $summary,
                'user' => $log->user?->name ?? 'System',
                'logged_at' => optional($log->logged_at)->toIso8601String(),
                'logged_at_label' => optional($log->logged_at)->diffForHumans(),
            ];
        });

        return response()->json([
            'items' => $items,
            'latest_id' => $latestId,
        ]);
    }

    public function create()
    {
        return view('admin.activity-logs.create', [
            'actions' => config('system.activity_types', []),
            'users' => User::query()->orderBy('name')->limit(200)->get(),
        ]);
    }

    public function store(StoreActivityLogRequest $request)
    {
        $data = $request->validated();

        ActivityLog::create([
            'user_id' => $data['user_id'] ?? null,
            'action' => $data['action'],
            'subject_type' => $data['subject_type'] ?? null,
            'subject_id' => $data['subject_id'] ?? null,
            'description' => $data['description'] ?? null,
            'ip_address' => $data['ip_address'] ?? null,
            'user_agent' => $data['user_agent'] ?? null,
            'logged_at' => $this->parseDateTime($data['logged_at'] ?? null) ?? now(),
        ]);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.activity-logs.create')
                ->with('status', 'Activity log created. You can add another.');
        }

        return redirect()
            ->route('admin.activity-logs.index')
            ->with('status', 'Activity log created successfully.');
    }

    public function show(ActivityLog $activityLog)
    {
        $activityLog->load('user');

        return view('admin.activity-logs.show', [
            'log' => $activityLog,
            'actions' => config('system.activity_types', []),
        ]);
    }

    public function edit(ActivityLog $activityLog)
    {
        return view('admin.activity-logs.edit', [
            'log' => $activityLog,
            'actions' => config('system.activity_types', []),
            'users' => User::query()->orderBy('name')->limit(200)->get(),
        ]);
    }

    public function update(UpdateActivityLogRequest $request, ActivityLog $activityLog)
    {
        $data = $request->validated();

        $activityLog->update([
            'user_id' => $data['user_id'] ?? null,
            'action' => $data['action'],
            'subject_type' => $data['subject_type'] ?? null,
            'subject_id' => $data['subject_id'] ?? null,
            'description' => $data['description'] ?? null,
            'ip_address' => $data['ip_address'] ?? null,
            'user_agent' => $data['user_agent'] ?? null,
            'logged_at' => $this->parseDateTime($data['logged_at'] ?? null),
        ]);

        return redirect()
            ->route('admin.activity-logs.edit', $activityLog)
            ->with('status', 'Activity log updated successfully.');
    }

    public function destroy(ActivityLog $activityLog)
    {
        $activityLog->delete();

        return redirect()
            ->route('admin.activity-logs.index')
            ->with('status', 'Activity log deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:activity_logs,id'],
        ]);

        ActivityLog::query()->whereIn('id', $validated['ids'])->delete();

        return redirect()
            ->route('admin.activity-logs.index')
            ->with('status', 'Selected activity logs deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $action = $request->query('action');
        $userId = $request->query('user');
        $sort = $request->query('sort', 'latest');

        $query = ActivityLog::query()->with('user');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('description', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('subject_type', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('action', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($action) {
            $query->where('action', $action);
        }

        if ($userId) {
            $query->where('user_id', $userId);
        }

        $this->applySort($query, $sort);

        $filename = 'activity-logs-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, [
                'user_email',
                'action',
                'subject_type',
                'subject_id',
                'description',
                'ip_address',
                'user_agent',
                'logged_at',
            ]);

            $query->chunk(200, function ($logs) use ($handle) {
                foreach ($logs as $log) {
                    fputcsv($handle, [
                        $log->user?->email,
                        $log->action,
                        $log->subject_type,
                        $log->subject_id,
                        $log->description,
                        $log->ip_address,
                        $log->user_agent,
                        optional($log->logged_at)->format('Y-m-d H:i:s'),
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportActivityLogsRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.activity-logs.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.activity-logs.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, [
            'user_email',
            'action',
            'subject_type',
            'subject_id',
            'description',
            'ip_address',
            'user_agent',
            'logged_at',
        ]);

        $actions = array_keys(config('system.activity_types', []));

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $action = $this->normalizeAction($data['action'] ?? null, $actions);

            if ($action === '') {
                $skipped++;
                continue;
            }

            $userId = null;
            $userEmail = trim((string) ($data['user_email'] ?? ''));
            if ($userEmail !== '') {
                $userId = User::query()->where('email', $userEmail)->value('id');
            }

            ActivityLog::create([
                'user_id' => $userId,
                'action' => $action,
                'subject_type' => $data['subject_type'] ?? null,
                'subject_id' => $this->parseInt($data['subject_id'] ?? null),
                'description' => $data['description'] ?? null,
                'ip_address' => $data['ip_address'] ?? null,
                'user_agent' => $data['user_agent'] ?? null,
                'logged_at' => $this->parseDateTime($data['logged_at'] ?? null) ?? now(),
            ]);

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.activity-logs.index')
            ->with('status', "Imported {$imported} activity logs. Skipped {$skipped} rows.");
    }

    private function parseDateTime(?string $value): ?Carbon
    {
        if (! $value) {
            return null;
        }

        try {
            return Carbon::parse($value);
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function parseInt($value): ?int
    {
        if ($value === null || trim((string) $value) === '') {
            return null;
        }

        return (int) $value;
    }

    private function normalizeAction(?string $value, array $actions): string
    {
        $value = Str::lower(trim((string) $value));

        return in_array($value, $actions, true) ? $value : '';
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'oldest' => $query->orderBy('logged_at', 'asc'),
            'action_asc' => $query->orderBy('action'),
            'action_desc' => $query->orderBy('action', 'desc'),
            default => $query->orderBy('logged_at', 'desc'),
        };
    }
}
