<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ActivityLog;
use App\Models\Event;
use App\Models\EventMedia;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Schema;

class AdminSearchController extends Controller
{
    public function __invoke(Request $request)
    {
        $query = trim((string) $request->query('q', ''));
        $limit = (int) $request->query('limit', 5);
        $limit = max(1, min($limit, 10));

        if ($query === '' || mb_strlen($query) < 2) {
            return response()->json([
                'query' => $query,
                'items' => [],
            ]);
        }

        $items = [];

        if (Gate::allows('viewAny', User::class) && Schema::hasTable('users')) {
            $users = User::query()
                ->where(function ($builder) use ($query) {
                    $builder->where('name', 'like', "%{$query}%")
                        ->orWhere('email', 'like', "%{$query}%");
                })
                ->orderBy('name')
                ->limit($limit)
                ->get(['id', 'name', 'email']);

            foreach ($users as $user) {
                $items[] = [
                    'type' => 'Users',
                    'title' => $user->name,
                    'subtitle' => $user->email,
                    'url' => route('admin.users.edit', $user),
                ];
            }
        }

        if (Gate::allows('viewAny', Event::class) && Schema::hasTable('events')) {
            $events = Event::query()
                ->where(function ($builder) use ($query) {
                    $builder->where('name', 'like', "%{$query}%")
                        ->orWhere('slug', 'like', "%{$query}%")
                        ->orWhere('type', 'like', "%{$query}%");
                })
                ->orderByDesc('event_date')
                ->limit($limit)
                ->get(['id', 'name', 'type', 'slug']);

            foreach ($events as $event) {
                $items[] = [
                    'type' => 'Events',
                    'title' => $event->name,
                    'subtitle' => $event->type ?: 'Event',
                    'url' => route('admin.events.show', $event),
                ];
            }
        }

        if (Gate::allows('viewAny', EventMedia::class) && Schema::hasTable('event_media')) {
            $media = EventMedia::query()
                ->with('event:id,name,slug')
                ->where(function ($builder) use ($query) {
                    $builder->where('file_name', 'like', "%{$query}%")
                        ->orWhere('mime_type', 'like', "%{$query}%");
                })
                ->orderByDesc('created_at')
                ->limit($limit)
                ->get(['id', 'event_id', 'file_name', 'mime_type']);

            foreach ($media as $item) {
                $event = $item->event;
                $items[] = [
                    'type' => 'Uploads',
                    'title' => $item->file_name ?: 'Upload',
                    'subtitle' => $event?->name ?? 'Event media',
                    'url' => $event ? route('admin.events.show', $event) : route('admin.events.index'),
                ];
            }
        }

        if (Gate::allows('viewAny', ActivityLog::class) && Schema::hasTable('activity_logs')) {
            $logs = ActivityLog::query()
                ->where(function ($builder) use ($query) {
                    $builder->where('description', 'like', "%{$query}%")
                        ->orWhere('action', 'like', "%{$query}%");
                })
                ->orderByDesc('logged_at')
                ->limit($limit)
                ->get(['id', 'description', 'action']);

            foreach ($logs as $log) {
                $items[] = [
                    'type' => 'Activity Logs',
                    'title' => $log->description ?: ucfirst($log->action),
                    'subtitle' => ucfirst($log->action),
                    'url' => route('admin.activity-logs.show', $log),
                ];
            }
        }

        return response()->json([
            'query' => $query,
            'items' => $items,
        ]);
    }
}
