<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportBlockedIpsRequest;
use App\Http\Requests\Admin\StoreBlockedIpRequest;
use App\Http\Requests\Admin\UpdateBlockedIpRequest;
use App\Models\BlockedIp;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class BlockedIpController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = BlockedIp::query()->with('creator');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('ip_address', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('reason', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return view('admin.security.blocked-ips.index', [
            'blockedIps' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'status' => $status,
            'sort' => $sort,
        ]);
    }

    public function create()
    {
        return view('admin.security.blocked-ips.create');
    }

    public function store(StoreBlockedIpRequest $request)
    {
        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');
        $data['created_by'] = $request->user()?->id;

        BlockedIp::create($data);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.blocked-ips.create')
                ->with('status', 'Blocked IP created. You can add another.');
        }

        return redirect()
            ->route('admin.blocked-ips.index')
            ->with('status', 'Blocked IP created successfully.');
    }

    public function show(BlockedIp $blockedIp)
    {
        $blockedIp->load('creator');

        return view('admin.security.blocked-ips.show', [
            'blockedIp' => $blockedIp,
        ]);
    }

    public function edit(BlockedIp $blockedIp)
    {
        return view('admin.security.blocked-ips.edit', [
            'blockedIp' => $blockedIp,
        ]);
    }

    public function update(UpdateBlockedIpRequest $request, BlockedIp $blockedIp)
    {
        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        $blockedIp->update($data);

        return redirect()
            ->route('admin.blocked-ips.edit', $blockedIp)
            ->with('status', 'Blocked IP updated successfully.');
    }

    public function destroy(BlockedIp $blockedIp)
    {
        $blockedIp->delete();

        return redirect()
            ->route('admin.blocked-ips.index')
            ->with('status', 'Blocked IP deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:blocked_ips,id'],
        ]);

        BlockedIp::query()->whereIn('id', $validated['ids'])->delete();

        return redirect()
            ->route('admin.blocked-ips.index')
            ->with('status', 'Selected blocked IPs deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = BlockedIp::query()->with('creator');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('ip_address', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('reason', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $filename = 'blocked-ips-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['ip_address', 'reason', 'is_active', 'created_by', 'created_at']);

            $query->chunk(200, function ($blockedIps) use ($handle) {
                foreach ($blockedIps as $blockedIp) {
                    fputcsv($handle, [
                        $blockedIp->ip_address,
                        $blockedIp->reason,
                        $blockedIp->is_active ? 1 : 0,
                        $blockedIp->creator?->email,
                        optional($blockedIp->created_at)->format('Y-m-d H:i:s'),
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportBlockedIpsRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.blocked-ips.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.blocked-ips.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, ['ip_address', 'reason', 'is_active']);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $ipAddress = trim((string) ($data['ip_address'] ?? ''));

            if ($ipAddress === '') {
                $skipped++;
                continue;
            }

            BlockedIp::updateOrCreate(
                ['ip_address' => $ipAddress],
                [
                    'reason' => $data['reason'] ?? null,
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.blocked-ips.index')
            ->with('status', "Imported {$imported} blocked IPs. Skipped {$skipped} rows.");
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'ip_asc' => $query->orderBy('ip_address'),
            'ip_desc' => $query->orderBy('ip_address', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }
}
