<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportBlockedLocationsRequest;
use App\Http\Requests\Admin\StoreBlockedLocationRequest;
use App\Http\Requests\Admin\UpdateBlockedLocationRequest;
use App\Models\BlockedLocation;
use App\Models\City;
use App\Models\Country;
use App\Models\State;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class BlockedLocationController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $countryId = $request->query('country_id');
        $stateId = $request->query('state_id');
        $cityId = $request->query('city_id');
        $sort = $request->query('sort', 'latest');

        $query = BlockedLocation::query()->with(['country', 'state', 'city', 'creator']);

        if ($search) {
            $query->where('reason', 'like', "%" . $this->escapeLike($search) . "%");
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        if ($countryId) {
            $query->where('country_id', $countryId);
        }

        if ($stateId) {
            $query->where('state_id', $stateId);
        }

        if ($cityId) {
            $query->where('city_id', $cityId);
        }

        $this->applySort($query, $sort);

        return view('admin.security.blocked-locations.index', [
            'blockedLocations' => $query->paginate(10)->withQueryString(),
            'countries' => Country::query()->orderBy('name')->get(),
            'states' => State::query()->orderBy('name')->get(),
            'cities' => City::query()->orderBy('name')->get(),
            'search' => $search,
            'status' => $status,
            'countryId' => $countryId,
            'stateId' => $stateId,
            'cityId' => $cityId,
            'sort' => $sort,
        ]);
    }

    public function create()
    {
        return view('admin.security.blocked-locations.create', [
            'countries' => Country::query()->orderBy('name')->get(),
            'states' => State::query()->orderBy('name')->get(),
            'cities' => City::query()->orderBy('name')->get(),
        ]);
    }

    public function store(StoreBlockedLocationRequest $request)
    {
        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');
        $data['created_by'] = $request->user()?->id;

        BlockedLocation::create($data);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.blocked-locations.create')
                ->with('status', 'Blocked location created. You can add another.');
        }

        return redirect()
            ->route('admin.blocked-locations.index')
            ->with('status', 'Blocked location created successfully.');
    }

    public function show(BlockedLocation $blockedLocation)
    {
        $blockedLocation->load(['country', 'state', 'city', 'creator']);

        return view('admin.security.blocked-locations.show', [
            'blockedLocation' => $blockedLocation,
        ]);
    }

    public function edit(BlockedLocation $blockedLocation)
    {
        return view('admin.security.blocked-locations.edit', [
            'blockedLocation' => $blockedLocation,
            'countries' => Country::query()->orderBy('name')->get(),
            'states' => State::query()->orderBy('name')->get(),
            'cities' => City::query()->orderBy('name')->get(),
        ]);
    }

    public function update(UpdateBlockedLocationRequest $request, BlockedLocation $blockedLocation)
    {
        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        $blockedLocation->update($data);

        return redirect()
            ->route('admin.blocked-locations.edit', $blockedLocation)
            ->with('status', 'Blocked location updated successfully.');
    }

    public function destroy(BlockedLocation $blockedLocation)
    {
        $blockedLocation->delete();

        return redirect()
            ->route('admin.blocked-locations.index')
            ->with('status', 'Blocked location deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:blocked_locations,id'],
        ]);

        BlockedLocation::query()->whereIn('id', $validated['ids'])->delete();

        return redirect()
            ->route('admin.blocked-locations.index')
            ->with('status', 'Selected blocked locations deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $countryId = $request->query('country_id');
        $stateId = $request->query('state_id');
        $cityId = $request->query('city_id');
        $sort = $request->query('sort', 'latest');

        $query = BlockedLocation::query()->with(['country', 'state', 'city', 'creator']);

        if ($search) {
            $query->where('reason', 'like', "%" . $this->escapeLike($search) . "%");
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        if ($countryId) {
            $query->where('country_id', $countryId);
        }

        if ($stateId) {
            $query->where('state_id', $stateId);
        }

        if ($cityId) {
            $query->where('city_id', $cityId);
        }

        $this->applySort($query, $sort);

        $filename = 'blocked-locations-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['country', 'state', 'city', 'reason', 'is_active', 'created_by', 'created_at']);

            $query->chunk(200, function ($blockedLocations) use ($handle) {
                foreach ($blockedLocations as $blockedLocation) {
                    fputcsv($handle, [
                        $blockedLocation->country?->name,
                        $blockedLocation->state?->name,
                        $blockedLocation->city?->name,
                        $blockedLocation->reason,
                        $blockedLocation->is_active ? 1 : 0,
                        $blockedLocation->creator?->email,
                        optional($blockedLocation->created_at)->format('Y-m-d H:i:s'),
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportBlockedLocationsRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.blocked-locations.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.blocked-locations.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, ['country', 'state', 'city', 'reason', 'is_active']);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $countryName = trim((string) ($data['country'] ?? ''));
            $stateName = trim((string) ($data['state'] ?? ''));
            $cityName = trim((string) ($data['city'] ?? ''));

            $countryId = $countryName !== ''
                ? Country::query()->where('name', $countryName)->value('id')
                : null;
            $stateId = $stateName !== ''
                ? State::query()->where('name', $stateName)->value('id')
                : null;
            $cityId = $cityName !== ''
                ? City::query()->where('name', $cityName)->value('id')
                : null;

            if (! $countryId && ! $stateId && ! $cityId) {
                $skipped++;
                continue;
            }

            BlockedLocation::updateOrCreate(
                [
                    'country_id' => $countryId,
                    'state_id' => $stateId,
                    'city_id' => $cityId,
                ],
                [
                    'reason' => $data['reason'] ?? null,
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.blocked-locations.index')
            ->with('status', "Imported {$imported} blocked locations. Skipped {$skipped} rows.");
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'location_asc' => $query->orderBy('country_id')->orderBy('state_id')->orderBy('city_id'),
            'location_desc' => $query->orderBy('country_id', 'desc')->orderBy('state_id', 'desc')->orderBy('city_id', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }
}
