<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportBlogCategoriesRequest;
use App\Http\Requests\Admin\StoreBlogCategoryRequest;
use App\Http\Requests\Admin\UpdateBlogCategoryRequest;
use App\Models\BlogCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class BlogCategoryController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = BlogCategory::query()->withCount('posts');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return view('admin.content.blog-categories.index', [
            'categories' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'status' => $status,
            'sort' => $sort,
        ]);
    }

    public function create()
    {
        return view('admin.content.blog-categories.create');
    }

    public function store(StoreBlogCategoryRequest $request)
    {
        $data = $request->validated();

        BlogCategory::create([
            'name' => $data['name'],
            'slug' => $this->uniqueSlug($data['name']),
            'is_active' => $request->boolean('is_active'),
            'created_by' => $request->user()?->id,
        ]);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.blog-categories.create')
                ->with('status', 'Category created. You can add another.');
        }

        return redirect()
            ->route('admin.blog-categories.index')
            ->with('status', 'Category created successfully.');
    }

    public function show(BlogCategory $blogCategory)
    {
        $blogCategory->loadCount('posts');

        return view('admin.content.blog-categories.show', [
            'category' => $blogCategory,
        ]);
    }

    public function edit(BlogCategory $blogCategory)
    {
        return view('admin.content.blog-categories.edit', [
            'category' => $blogCategory,
        ]);
    }

    public function update(UpdateBlogCategoryRequest $request, BlogCategory $blogCategory)
    {
        $data = $request->validated();

        $payload = [
            'name' => $data['name'],
            'is_active' => $request->boolean('is_active'),
        ];

        if ($data['name'] !== $blogCategory->name) {
            $payload['slug'] = $this->uniqueSlug($data['name'], $blogCategory->id);
        }

        $blogCategory->update($payload);

        return redirect()
            ->route('admin.blog-categories.edit', $blogCategory)
            ->with('status', 'Category updated successfully.');
    }

    public function destroy(BlogCategory $blogCategory)
    {
        $blogCategory->delete();

        return redirect()
            ->route('admin.blog-categories.index')
            ->with('status', 'Category deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:blog_categories,id'],
        ]);

        BlogCategory::query()->whereIn('id', $validated['ids'])->delete();

        return redirect()
            ->route('admin.blog-categories.index')
            ->with('status', 'Selected categories deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = BlogCategory::query()->withCount('posts');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $filename = 'blog-categories-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['name', 'slug', 'posts_count', 'is_active']);

            $query->chunk(200, function ($categories) use ($handle) {
                foreach ($categories as $category) {
                    fputcsv($handle, [
                        $category->name,
                        $category->slug,
                        $category->posts_count,
                        $category->is_active ? 1 : 0,
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportBlogCategoriesRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.blog-categories.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.blog-categories.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, ['name', 'is_active']);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ''));

            if ($name === '') {
                $skipped++;
                continue;
            }

            $category = BlogCategory::query()->where('name', $name)->first();

            if ($category) {
                $category->update([
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]);
            } else {
                BlogCategory::create([
                    'name' => $name,
                    'slug' => $this->uniqueSlug($name),
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]);
            }

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.blog-categories.index')
            ->with('status', "Imported {$imported} categories. Skipped {$skipped} rows.");
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_desc' => $query->orderBy('name', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('name'),
        };
    }

    private function uniqueSlug(string $name, ?int $ignoreId = null): string
    {
        $base = Str::slug($name);
        $slug = $base !== '' ? $base : Str::random(8);
        $counter = 1;

        while ($this->slugExists($slug, $ignoreId)) {
            $slug = $base . '-' . $counter;
            $counter++;
        }

        return $slug;
    }

    private function slugExists(string $slug, ?int $ignoreId): bool
    {
        $query = BlogCategory::query()->where('slug', $slug);
        if ($ignoreId) {
            $query->where('id', '!=', $ignoreId);
        }

        return $query->exists();
    }
}
