<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportCitiesRequest;
use App\Http\Requests\Admin\StoreCityRequest;
use App\Http\Requests\Admin\UpdateCityRequest;
use App\Models\City;
use App\Models\Country;
use App\Models\State;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class CityController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $countryFilter = $request->query('country');
        $stateFilter = $request->query('state');
        $sort = $request->query('sort', 'name_asc');

        $query = City::query()->with(['state.country']);

        if ($search) {
            $query->where('name', 'like', "%" . $this->escapeLike($search) . "%");
        }

        if ($stateFilter) {
            $query->where('state_id', $stateFilter);
        }

        if ($countryFilter) {
            $query->whereHas('state', function ($builder) use ($countryFilter) {
                $builder->where('country_id', $countryFilter);
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return view('admin.locations.cities.index', [
            'cities' => $query->paginate(10)->withQueryString(),
            'countries' => Country::query()->orderBy('name')->get(),
            'states' => State::query()->orderBy('name')->get(),
            'search' => $search,
            'status' => $status,
            'countryFilter' => $countryFilter,
            'stateFilter' => $stateFilter,
            'sort' => $sort,
        ]);
    }

    public function create()
    {
        return view('admin.locations.cities.create', [
            'countries' => Country::query()->orderBy('name')->get(),
            'states' => State::query()->orderBy('name')->get(),
        ]);
    }

    public function store(StoreCityRequest $request)
    {
        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        City::create($data);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.cities.create')
                ->with('status', 'City created. You can add another.');
        }

        return redirect()
            ->route('admin.cities.index')
            ->with('status', 'City created successfully.');
    }

    public function show(City $city)
    {
        $city->load(['state.country']);

        return view('admin.locations.cities.show', [
            'city' => $city,
        ]);
    }

    public function edit(City $city)
    {
        return view('admin.locations.cities.edit', [
            'city' => $city,
            'countries' => Country::query()->orderBy('name')->get(),
            'states' => State::query()->orderBy('name')->get(),
        ]);
    }

    public function update(UpdateCityRequest $request, City $city)
    {
        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        $city->update($data);

        return redirect()
            ->route('admin.cities.edit', $city)
            ->with('status', 'City updated successfully.');
    }

    public function destroy(City $city)
    {
        $city->delete();

        return redirect()
            ->route('admin.cities.index')
            ->with('status', 'City deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:cities,id'],
        ]);

        City::query()->whereIn('id', $validated['ids'])->delete();

        return redirect()
            ->route('admin.cities.index')
            ->with('status', 'Selected cities deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $countryFilter = $request->query('country');
        $stateFilter = $request->query('state');
        $sort = $request->query('sort', 'name_asc');

        $query = City::query()->with(['state.country']);

        if ($search) {
            $query->where('name', 'like', "%" . $this->escapeLike($search) . "%");
        }

        if ($stateFilter) {
            $query->where('state_id', $stateFilter);
        }

        if ($countryFilter) {
            $query->whereHas('state', function ($builder) use ($countryFilter) {
                $builder->where('country_id', $countryFilter);
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $filename = 'cities-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['country', 'country_iso2', 'state', 'state_code', 'city', 'is_active']);

            $query->chunk(200, function ($cities) use ($handle) {
                foreach ($cities as $city) {
                    fputcsv($handle, [
                        $city->state?->country?->name,
                        $city->state?->country?->iso2,
                        $city->state?->name,
                        $city->state?->code,
                        $city->name,
                        $city->is_active ? 1 : 0,
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportCitiesRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.cities.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.cities.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, [
            'country',
            'country_iso2',
            'state',
            'state_name',
            'state_code',
            'code',
            'city',
            'name',
            'is_active',
        ]);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $stateId = $this->resolveStateId($data);
            $cityName = trim((string) ($data['city'] ?? $data['name'] ?? ''));

            if (! $stateId || $cityName === '') {
                $skipped++;
                continue;
            }

            City::updateOrCreate(
                [
                    'state_id' => $stateId,
                    'name' => $cityName,
                ],
                [
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.cities.index')
            ->with('status', "Imported {$imported} cities. Skipped {$skipped} rows.");
    }

    private function resolveStateId(array $data): ?int
    {
        $stateName = trim((string) ($data['state'] ?? $data['state_name'] ?? ''));
        $stateCode = trim((string) ($data['state_code'] ?? $data['code'] ?? ''));

        if ($stateName === '' && $stateCode === '') {
            return null;
        }

        $query = State::query();
        $countryId = $this->resolveCountryId($data);

        if ($countryId) {
            $query->where('country_id', $countryId);
        }

        if ($stateCode !== '') {
            $query->where('code', $stateCode);
        }

        if ($stateName !== '') {
            $query->where('name', $stateName);
        }

        return $query->value('id');
    }

    private function resolveCountryId(array $data): ?int
    {
        $countryIso2 = trim((string) ($data['country_iso2'] ?? ''));
        if ($countryIso2 !== '') {
            return Country::query()
                ->where('iso2', strtoupper($countryIso2))
                ->value('id');
        }

        $countryName = trim((string) ($data['country'] ?? ''));
        if ($countryName !== '') {
            return Country::query()
                ->where('name', $countryName)
                ->value('id');
        }

        return null;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_desc' => $query->orderBy('name', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('name'),
        };
    }
}
