<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportCountriesRequest;
use App\Http\Requests\Admin\StoreCountryRequest;
use App\Http\Requests\Admin\UpdateCountryRequest;
use App\Models\City;
use App\Models\Country;
use App\Models\State;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class CountryController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = Country::query()->withCount('states');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('iso2', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('iso3', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return view('admin.locations.countries.index', [
            'countries' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'status' => $status,
            'sort' => $sort,
        ]);
    }

    public function create()
    {
        return view('admin.locations.countries.create');
    }

    public function store(StoreCountryRequest $request)
    {
        $data = $request->validated();
        $autoImport = $request->boolean('auto_import');

        unset($data['auto_import']);

        $data['iso2'] = $this->normalizeCode($data['iso2'] ?? null);
        $data['iso3'] = $this->normalizeCode($data['iso3'] ?? null);
        $data['phone_code'] = $this->normalizeCode($data['phone_code'] ?? null);
        $data['is_active'] = $request->boolean('is_active');

        $country = Country::create($data);

        $importSummary = $autoImport ? $this->importLocationData($country) : null;
        $message = $importSummary
            ? "Country created. Imported {$importSummary['states']} states and {$importSummary['cities']} cities."
            : 'Country created successfully.';

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.countries.create')
                ->with('status', $message);
        }

        return redirect()
            ->route('admin.countries.index')
            ->with('status', $message);
    }

    public function show(Country $country)
    {
        $country->load(['states' => function ($query) {
            $query->orderBy('name');
        }]);

        return view('admin.locations.countries.show', [
            'country' => $country,
        ]);
    }

    public function edit(Country $country)
    {
        return view('admin.locations.countries.edit', [
            'country' => $country,
        ]);
    }

    public function update(UpdateCountryRequest $request, Country $country)
    {
        $data = $request->validated();
        $autoImport = $request->boolean('auto_import');

        unset($data['auto_import']);

        $data['iso2'] = $this->normalizeCode($data['iso2'] ?? null);
        $data['iso3'] = $this->normalizeCode($data['iso3'] ?? null);
        $data['phone_code'] = $this->normalizeCode($data['phone_code'] ?? null);
        $data['is_active'] = $request->boolean('is_active');

        $country->update($data);

        $importSummary = $autoImport ? $this->importLocationData($country) : null;

        if ($importSummary) {
            return redirect()
                ->route('admin.countries.edit', $country)
                ->with('status', "Country updated. Imported {$importSummary['states']} states and {$importSummary['cities']} cities.");
        }

        return redirect()
            ->route('admin.countries.edit', $country)
            ->with('status', 'Country updated successfully.');
    }

    public function destroy(Country $country)
    {
        $country->delete();

        return redirect()
            ->route('admin.countries.index')
            ->with('status', 'Country deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:countries,id'],
        ]);

        Country::query()->whereIn('id', $validated['ids'])->delete();

        return redirect()
            ->route('admin.countries.index')
            ->with('status', 'Selected countries deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = Country::query()->withCount('states');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('iso2', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('iso3', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $filename = 'countries-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['name', 'iso2', 'iso3', 'phone_code', 'states_count', 'is_active']);

            $query->orderBy('name')->chunk(200, function ($countries) use ($handle) {
                foreach ($countries as $country) {
                    fputcsv($handle, [
                        $country->name,
                        $country->iso2,
                        $country->iso3,
                        $country->phone_code,
                        $country->states_count,
                        $country->is_active ? 1 : 0,
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportCountriesRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.countries.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.countries.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, ['name', 'iso2', 'iso3', 'phone_code', 'is_active']);

        if (! in_array('name', $map, true)) {
            fclose($handle);
            return redirect()
                ->route('admin.countries.index')
                ->with('error', 'CSV must include a name column.');
        }

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ''));

            if ($name === '') {
                $skipped++;
                continue;
            }

            Country::updateOrCreate(
                ['name' => $name],
                [
                    'iso2' => $this->normalizeCode($data['iso2'] ?? null),
                    'iso3' => $this->normalizeCode($data['iso3'] ?? null),
                    'phone_code' => $this->normalizeCode($data['phone_code'] ?? null),
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.countries.index')
            ->with('status', "Imported {$imported} countries. Skipped {$skipped} rows.");
    }

    private function normalizeCode(?string $value): ?string
    {
        if ($value === null) {
            return null;
        }

        $value = strtoupper(trim($value));

        return $value !== '' ? $value : null;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_desc' => $query->orderBy('name', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('name'),
        };
    }

    private function importLocationData(Country $country): ?array
    {
        $csvSummary = $this->importLocationDataFromCsv($country);
        if ($csvSummary) {
            return $csvSummary;
        }

        $dataset = config('locations.countries', []);
        $payload = null;

        if ($country->iso2 && isset($dataset[$country->iso2])) {
            $payload = $dataset[$country->iso2];
        } else {
            foreach ($dataset as $entry) {
                if (isset($entry['name']) && Str::lower($entry['name']) === Str::lower($country->name)) {
                    $payload = $entry;
                    break;
                }
            }
        }

        if (! $payload || empty($payload['states'])) {
            return null;
        }

        $statesCreated = 0;
        $citiesCreated = 0;

        foreach ($payload['states'] as $stateName => $cities) {
            $state = State::firstOrCreate(
                ['country_id' => $country->id, 'name' => $stateName],
                ['is_active' => true]
            );

            $statesCreated++;

            foreach ((array) $cities as $cityName) {
                City::firstOrCreate(
                    ['state_id' => $state->id, 'name' => $cityName],
                    ['is_active' => true]
                );
                $citiesCreated++;
            }
        }

        return [
            'states' => $statesCreated,
            'cities' => $citiesCreated,
        ];
    }

    private function importLocationDataFromCsv(Country $country): ?array
    {
        $statesPath = public_path('templates/states.csv');
        $citiesPath = public_path('templates/cities.csv');

        if (! is_file($statesPath) && ! is_file($citiesPath)) {
            return null;
        }

        $statesCreated = 0;
        $citiesCreated = 0;

        $statesByName = [];
        $statesByCode = [];

        $existingStates = State::query()
            ->where('country_id', $country->id)
            ->get(['id', 'name', 'code']);

        foreach ($existingStates as $state) {
            $nameKey = Str::lower($state->name);
            if ($nameKey !== '') {
                $statesByName[$nameKey] = $state;
            }
            if ($state->code) {
                $statesByCode[Str::upper($state->code)] = $state;
            }
        }

        if (is_file($statesPath)) {
            $rows = $this->readCsvRows($statesPath, ['country', 'country_iso2', 'iso2', 'state', 'code', 'is_active']);

            foreach ($rows as $data) {
                if (! $this->matchesCountry($country, $data)) {
                    continue;
                }

                $stateName = trim((string) ($data['state'] ?? ''));
                if ($stateName === '') {
                    continue;
                }

                $stateCode = $this->normalizeCode($data['code'] ?? null);
                $isActive = $this->parseBool($data['is_active'] ?? null, true);

                $state = State::updateOrCreate(
                    ['country_id' => $country->id, 'name' => $stateName],
                    ['code' => $stateCode, 'is_active' => $isActive]
                );

                if ($state->wasRecentlyCreated) {
                    $statesCreated++;
                }

                $statesByName[Str::lower($stateName)] = $state;
                if ($stateCode) {
                    $statesByCode[Str::upper($stateCode)] = $state;
                }
            }
        }

        if (is_file($citiesPath)) {
            $rows = $this->readCsvRows($citiesPath, ['country', 'country_iso2', 'iso2', 'state', 'state_code', 'city', 'is_active']);

            foreach ($rows as $data) {
                if (! $this->matchesCountry($country, $data)) {
                    continue;
                }

                $cityName = trim((string) ($data['city'] ?? ''));
                if ($cityName === '') {
                    continue;
                }

                $stateName = trim((string) ($data['state'] ?? ''));
                $stateCode = $this->normalizeCode($data['state_code'] ?? null);
                $state = $this->resolveStateForImport($country, $stateName, $stateCode, $statesByName, $statesByCode);

                if (! $state && $stateName !== '') {
                    $state = State::updateOrCreate(
                        ['country_id' => $country->id, 'name' => $stateName],
                        ['code' => $stateCode, 'is_active' => true]
                    );

                    if ($state->wasRecentlyCreated) {
                        $statesCreated++;
                    }

                    $statesByName[Str::lower($stateName)] = $state;
                    if ($stateCode) {
                        $statesByCode[Str::upper($stateCode)] = $state;
                    }
                }

                if (! $state) {
                    continue;
                }

                $city = City::updateOrCreate(
                    ['state_id' => $state->id, 'name' => $cityName],
                    ['is_active' => $this->parseBool($data['is_active'] ?? null, true)]
                );

                if ($city->wasRecentlyCreated) {
                    $citiesCreated++;
                }
            }
        }

        if ($statesCreated === 0 && $citiesCreated === 0) {
            return null;
        }

        return [
            'states' => $statesCreated,
            'cities' => $citiesCreated,
        ];
    }

    private function matchesCountry(Country $country, array $data): bool
    {
        $countryName = Str::lower(trim((string) ($data['country'] ?? '')));
        $countryIso2 = strtoupper(trim((string) ($data['country_iso2'] ?? ($data['iso2'] ?? ''))));

        if ($countryIso2 !== '' && $country->iso2) {
            return $countryIso2 === strtoupper($country->iso2);
        }

        if ($countryName !== '') {
            return $countryName === Str::lower($country->name);
        }

        return false;
    }

    private function resolveStateForImport(Country $country, string $stateName, ?string $stateCode, array $statesByName, array $statesByCode): ?State
    {
        if ($stateCode) {
            $state = $statesByCode[Str::upper($stateCode)] ?? null;
            if ($state) {
                return $state;
            }

            $state = State::query()
                ->where('country_id', $country->id)
                ->where('code', $stateCode)
                ->first();

            if ($state) {
                return $state;
            }
        }

        if ($stateName !== '') {
            $state = $statesByName[Str::lower($stateName)] ?? null;
            if ($state) {
                return $state;
            }

            return State::query()
                ->where('country_id', $country->id)
                ->where('name', $stateName)
                ->first();
        }

        return null;
    }

    private function readCsvRows(string $path, array $allowed): array
    {
        $handle = fopen($path, 'r');
        if (! $handle) {
            return [];
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return [];
        }

        $map = $this->mapHeaders($header, $allowed);
        if ($map === []) {
            fclose($handle);
            return [];
        }

        $rows = [];
        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $rows[] = $this->extractRow($row, $map);
        }

        fclose($handle);

        return $rows;
    }
}
