<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportCouponsRequest;
use App\Http\Requests\Admin\StoreCouponRequest;
use App\Http\Requests\Admin\UpdateCouponRequest;
use App\Models\Coupon;
use App\Models\Plan;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;

class CouponController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $type = $request->query('type');
        $sort = $request->query('sort', 'latest');

        $query = Coupon::query()->withCount('plans');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('code', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($type) {
            $query->where('discount_type', $type);
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return view('admin.coupons.index', [
            'coupons' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'status' => $status,
            'typeFilter' => $type,
            'sort' => $sort,
        ]);
    }

    public function create()
    {
        return view('admin.coupons.create', [
            'plans' => Plan::query()->orderBy('name')->get(),
        ]);
    }

    public function store(StoreCouponRequest $request)
    {
        $data = $request->validated();

        $coupon = Coupon::create([
            'name' => $data['name'],
            'code' => $data['code'],
            'discount_type' => $data['discount_type'],
            'discount_value' => $data['discount_value'],
            'min_amount' => $data['min_amount'] ?? 0,
            'starts_at' => $data['starts_at'] ?? null,
            'ends_at' => $data['ends_at'] ?? null,
            'new_users_only' => $request->boolean('new_users_only'),
            'is_active' => $request->boolean('is_active'),
            'created_by' => $request->user()?->id,
        ]);

        $coupon->plans()->sync($data['plans']);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.coupons.create')
                ->with('status', 'Coupon created. You can add another.');
        }

        return redirect()
            ->route('admin.coupons.index')
            ->with('status', 'Coupon created successfully.');
    }

    public function show(Coupon $coupon)
    {
        $coupon->load('plans');

        return view('admin.coupons.show', [
            'coupon' => $coupon,
        ]);
    }

    public function edit(Coupon $coupon)
    {
        return view('admin.coupons.edit', [
            'coupon' => $coupon->load('plans'),
            'plans' => Plan::query()->orderBy('name')->get(),
        ]);
    }

    public function update(UpdateCouponRequest $request, Coupon $coupon)
    {
        $data = $request->validated();

        $coupon->update([
            'name' => $data['name'],
            'code' => $data['code'],
            'discount_type' => $data['discount_type'],
            'discount_value' => $data['discount_value'],
            'min_amount' => $data['min_amount'] ?? 0,
            'starts_at' => $data['starts_at'] ?? null,
            'ends_at' => $data['ends_at'] ?? null,
            'new_users_only' => $request->boolean('new_users_only'),
            'is_active' => $request->boolean('is_active'),
        ]);

        $coupon->plans()->sync($data['plans']);

        return redirect()
            ->route('admin.coupons.edit', $coupon)
            ->with('status', 'Coupon updated successfully.');
    }

    public function destroy(Coupon $coupon)
    {
        $coupon->plans()->detach();
        $coupon->delete();

        return redirect()
            ->route('admin.coupons.index')
            ->with('status', 'Coupon deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:coupons,id'],
        ]);

        $coupons = Coupon::query()->whereIn('id', $validated['ids'])->get();
        foreach ($coupons as $coupon) {
            $coupon->plans()->detach();
            $coupon->delete();
        }

        return redirect()
            ->route('admin.coupons.index')
            ->with('status', 'Selected coupons deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $type = $request->query('type');
        $sort = $request->query('sort', 'latest');

        $query = Coupon::query()->with('plans');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('code', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($type) {
            $query->where('discount_type', $type);
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $filename = 'coupons-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, [
                'name',
                'code',
                'discount_type',
                'discount_value',
                'min_amount',
                'starts_at',
                'ends_at',
                'new_users_only',
                'is_active',
                'plans',
            ]);

            $query->chunk(200, function ($coupons) use ($handle) {
                foreach ($coupons as $coupon) {
                    fputcsv($handle, [
                        $coupon->name,
                        $coupon->code,
                        $coupon->discount_type,
                        $coupon->discount_value,
                        $coupon->min_amount,
                        optional($coupon->starts_at)->format('Y-m-d'),
                        optional($coupon->ends_at)->format('Y-m-d'),
                        $coupon->new_users_only ? 1 : 0,
                        $coupon->is_active ? 1 : 0,
                        $coupon->plans->pluck('name')->implode('|'),
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportCouponsRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.coupons.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.coupons.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, [
            'name',
            'code',
            'discount_type',
            'discount_value',
            'min_amount',
            'starts_at',
            'ends_at',
            'new_users_only',
            'is_active',
            'plans',
        ]);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ''));
            $code = strtoupper(trim((string) ($data['code'] ?? '')));
            $discountType = $this->normalizeDiscountType($data['discount_type'] ?? null);
            $discountValue = $this->parseAmount($data['discount_value'] ?? null);

            if ($name === '' || $code === '' || ! $discountType || $discountValue === null) {
                $skipped++;
                continue;
            }

            if ($discountType === 'percent' && $discountValue > 100) {
                $skipped++;
                continue;
            }

            $planIds = $this->resolvePlanIds($data['plans'] ?? null);
            if (! $planIds) {
                $skipped++;
                continue;
            }

            $coupon = Coupon::query()->where('code', $code)->first();
            $payload = [
                'name' => $name,
                'code' => $code,
                'discount_type' => $discountType,
                'discount_value' => $discountValue,
                'min_amount' => $this->parseAmount($data['min_amount'] ?? null) ?? 0,
                'starts_at' => $this->parseDate($data['starts_at'] ?? null),
                'ends_at' => $this->parseDate($data['ends_at'] ?? null),
                'new_users_only' => $this->parseBool($data['new_users_only'] ?? null, false),
                'is_active' => $this->parseBool($data['is_active'] ?? null, true),
            ];

            if ($coupon) {
                $coupon->update($payload);
            } else {
                $coupon = Coupon::create($payload);
            }

            $coupon->plans()->sync($planIds);

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.coupons.index')
            ->with('status', "Imported {$imported} coupons. Skipped {$skipped} rows.");
    }

    private function resolvePlanIds(?string $value): array
    {
        if (! $value) {
            return [];
        }

        $parts = collect(preg_split('/[|,]/', $value))
            ->map(fn ($item) => trim((string) $item))
            ->filter()
            ->unique()
            ->values();

        $planIds = [];
        foreach ($parts as $entry) {
            if (is_numeric($entry)) {
                $id = Plan::query()->where('id', (int) $entry)->value('id');
                if ($id) {
                    $planIds[] = $id;
                }
                continue;
            }

            $id = Plan::query()
                ->where('name', $entry)
                ->orWhere('slug', Str::slug($entry))
                ->value('id');

            if ($id) {
                $planIds[] = $id;
            }
        }

        return array_values(array_unique($planIds));
    }

    private function normalizeDiscountType(?string $value): ?string
    {
        if (! $value) {
            return null;
        }

        $value = Str::lower(trim($value));

        if (in_array($value, ['percent', 'percentage', '%'], true)) {
            return 'percent';
        }

        if (in_array($value, ['fixed', 'flat', 'amount'], true)) {
            return 'fixed';
        }

        return null;
    }

    private function parseDate(?string $value): ?string
    {
        if (! $value) {
            return null;
        }

        try {
            return Carbon::parse($value)->format('Y-m-d');
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function parseAmount($value): ?float
    {
        if ($value === null || trim((string) $value) === '') {
            return null;
        }

        return (float) $value;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_asc' => $query->orderBy('name'),
            'name_desc' => $query->orderBy('name', 'desc'),
            'code_asc' => $query->orderBy('code'),
            'code_desc' => $query->orderBy('code', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }
}
