<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportCurrenciesRequest;
use App\Http\Requests\Admin\StoreCurrencyRequest;
use App\Http\Requests\Admin\UpdateCurrencyRequest;
use App\Models\Currency;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class CurrencyController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = Currency::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('code', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('symbol', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return view('admin.locations.currencies.index', [
            'currencies' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'status' => $status,
            'sort' => $sort,
        ]);
    }

    public function create()
    {
        return view('admin.locations.currencies.create');
    }

    public function store(StoreCurrencyRequest $request)
    {
        $data = $request->validated();

        Currency::create([
            'name' => $data['name'],
            'code' => $this->normalizeCode($data['code']),
            'symbol' => $this->normalizeSymbol($data['symbol'] ?? null),
            'precision' => $this->normalizePrecision($data['precision'] ?? null),
            'exchange_rate' => $this->normalizeExchangeRate($data['exchange_rate'] ?? null),
            'is_active' => $request->boolean('is_active'),
        ]);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.currencies.create')
                ->with('status', 'Currency created. You can add another.');
        }

        return redirect()
            ->route('admin.currencies.index')
            ->with('status', 'Currency created successfully.');
    }

    public function show(Currency $currency)
    {
        return view('admin.locations.currencies.show', [
            'currency' => $currency,
        ]);
    }

    public function edit(Currency $currency)
    {
        return view('admin.locations.currencies.edit', [
            'currency' => $currency,
        ]);
    }

    public function update(UpdateCurrencyRequest $request, Currency $currency)
    {
        $data = $request->validated();

        $currency->update([
            'name' => $data['name'],
            'code' => $this->normalizeCode($data['code']),
            'symbol' => $this->normalizeSymbol($data['symbol'] ?? null),
            'precision' => $this->normalizePrecision($data['precision'] ?? null),
            'exchange_rate' => $this->normalizeExchangeRate($data['exchange_rate'] ?? null),
            'is_active' => $request->boolean('is_active'),
        ]);

        return redirect()
            ->route('admin.currencies.edit', $currency)
            ->with('status', 'Currency updated successfully.');
    }

    public function destroy(Currency $currency)
    {
        $currency->delete();

        return redirect()
            ->route('admin.currencies.index')
            ->with('status', 'Currency deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:currencies,id'],
        ]);

        Currency::query()->whereIn('id', $validated['ids'])->delete();

        return redirect()
            ->route('admin.currencies.index')
            ->with('status', 'Selected currencies deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = Currency::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('code', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('symbol', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $filename = 'currencies-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['name', 'code', 'symbol', 'precision', 'is_active']);

            $query->chunk(200, function ($currencies) use ($handle) {
                foreach ($currencies as $currency) {
                    fputcsv($handle, [
                        $currency->name,
                        $currency->code,
                        $currency->symbol,
                        $currency->precision,
                        $currency->is_active ? 1 : 0,
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportCurrenciesRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.currencies.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.currencies.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, ['name', 'code', 'symbol', 'precision', 'is_active']);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ''));
            $code = $this->normalizeCode($data['code'] ?? null);

            if ($name === '' || ! $code) {
                $skipped++;
                continue;
            }

            Currency::updateOrCreate(
                ['code' => $code],
                [
                    'name' => $name,
                    'symbol' => $this->normalizeSymbol($data['symbol'] ?? null),
                    'precision' => $this->normalizePrecision($data['precision'] ?? null),
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.currencies.index')
            ->with('status', "Imported {$imported} currencies. Skipped {$skipped} rows.");
    }

    public function fetchRates(): JsonResponse
    {
        try {
            $response = Http::timeout(15)->get('https://open.er-api.com/v6/latest/USD');

            if (! $response->successful()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to fetch exchange rates. API returned status ' . $response->status(),
                ], 422);
            }

            $data = $response->json();

            if (($data['result'] ?? '') !== 'success' || empty($data['rates'])) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid response from exchange rate API.',
                ], 422);
            }

            $rates = $data['rates'];
            $updated = 0;

            Currency::all()->each(function (Currency $currency) use ($rates, &$updated) {
                if (isset($rates[$currency->code])) {
                    $currency->update(['exchange_rate' => $rates[$currency->code]]);
                    $updated++;
                }
            });

            return response()->json([
                'success' => true,
                'message' => "Updated exchange rates for {$updated} currencies.",
                'updated' => $updated,
                'source'  => 'open.er-api.com',
                'time'    => $data['time_last_update_utc'] ?? now()->toISOString(),
            ]);
        } catch (\Exception $e) {
            Log::error('Exchange rate fetch failed', ['error' => $e->getMessage()]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch exchange rates: ' . $e->getMessage(),
            ], 500);
        }
    }

    private function normalizeExchangeRate($value): float
    {
        if ($value === null || trim((string) $value) === '') {
            return 1.000000;
        }

        $rate = (float) $value;

        return $rate > 0 ? $rate : 1.000000;
    }

    private function normalizeCode(?string $value): ?string
    {
        if ($value === null) {
            return null;
        }

        $value = strtoupper(trim($value));

        return $value !== '' ? $value : null;
    }

    private function normalizeSymbol(?string $value): ?string
    {
        if ($value === null) {
            return null;
        }

        $value = trim($value);

        return $value !== '' ? $value : null;
    }

    private function normalizePrecision($value): int
    {
        if ($value === null || trim((string) $value) === '') {
            return 2;
        }

        $precision = (int) $value;

        if ($precision < 0) {
            return 0;
        }

        if ($precision > 6) {
            return 6;
        }

        return $precision;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_desc' => $query->orderBy('name', 'desc'),
            'code_asc' => $query->orderBy('code'),
            'code_desc' => $query->orderBy('code', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('name'),
        };
    }
}
