<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ActivityLog;
use App\Models\BlockedIp;
use App\Models\BlockedLocation;
use App\Models\Event;
use App\Models\EventMedia;
use App\Models\Order;
use App\Models\Plan;
use App\Models\Currency;
use App\Models\User;
use App\Services\StorageUsageService;
use App\Support\FormatSettings;
use App\Support\SecuritySettings;
use App\Support\SocialLoginSettings;

class DashboardController extends Controller
{
    public function index()
    {
        $user = auth()->user();
        $isAdmin = $user->hasRole(['Super Admin', 'Admin']);

        $eventQuery = Event::query();
        $mediaQuery = EventMedia::query();

        if (! $isAdmin) {
            $eventQuery->where('created_by', $user->id);
            $mediaQuery->whereHas('event', function ($q) use ($user) {
                $q->where('created_by', $user->id);
            });
        }

        $activeEventsCount = (clone $eventQuery)->where('is_active', true)->count();
        $eventsThisWeek = (clone $eventQuery)
            ->where('created_at', '>=', now()->subDays(7))
            ->count();

        $queueCount = (clone $mediaQuery)
            ->whereIn('status', ['pending', 'processing'])
            ->count();

        $storageSummary = app(StorageUsageService::class)->summary($isAdmin ? null : $user);
        $storagePercent = $storageSummary['percent'];
        $storageWarning = $storageSummary['warning'];

        $activeUsersCount = $isAdmin ? User::query()->where('is_active', true)->count() : 0;
        $newUsersThisWeek = $isAdmin ? User::query()->where('created_at', '>=', now()->subDays(7))->count() : 0;

        $eventHealth = (clone $eventQuery)
            ->withCount('media')
            ->withCount([
                'media as pending_media_count' => function ($query) {
                    $query->whereIn('status', ['pending', 'processing']);
                },
            ])
            ->orderByDesc('event_date')
            ->limit(3)
            ->get()
            ->map(function (Event $event) {
                $status = $event->pending_media_count > 0
                    ? 'Processing'
                    : ($event->is_active ? 'Live' : 'Archived');
                $subtitle = trim(($event->type ?: 'Event') . ' | ' . number_format($event->media_count) . ' assets');

                return [
                    'name' => $event->name,
                    'subtitle' => $subtitle,
                    'status' => $status,
                ];
            })
            ->all();

        $recentDeliveries = (clone $eventQuery)
            ->withCount('media')
            ->orderByDesc('updated_at')
            ->limit(3)
            ->get()
            ->map(function (Event $event) {
                $expiryLabel = 'No expiry';
                if ($event->expiry_date) {
                    $days = now()->diffInDays($event->expiry_date, false);
                    if ($days < 0) {
                        $expiryLabel = 'Expired ' . abs($days) . ' days ago';
                    } elseif ($days === 0) {
                        $expiryLabel = 'Expires today';
                    } else {
                        $expiryLabel = 'Expires in ' . $days . ' days';
                    }
                }

                return [
                    'name' => $event->name,
                    'type' => $event->type ?: 'Event',
                    'expiry' => $expiryLabel,
                    'status' => $event->guest_pin ? 'Guest pin' : 'Shared',
                    'assets' => $event->media_count,
                ];
            })
            ->all();

        $throughput = $this->buildThroughput($queueCount, $isAdmin ? null : $user);

        $twoFactorRequired = (bool) SecuritySettings::getValue('two_factor_admin_only', false);
        $socialProviders = SocialLoginSettings::enabledProviders();

        // Total media count
        $totalMediaCount = (clone $mediaQuery)->count();

        // Revenue stats (admin only)
        $currencyConfig = FormatSettings::currencyConfig();
        $revenueStats = null;
        if ($isAdmin) {
            // Load exchange rates for currency conversion (rates are relative to USD)
            $currencyRates = Currency::query()
                ->where('is_active', true)
                ->pluck('exchange_rate', 'code')
                ->mapWithKeys(fn ($rate, $code) => [strtoupper($code) => (float) $rate])
                ->all();

            $defaultCode = strtoupper($currencyConfig['code'] ?? 'USD');
            $defaultRate = (float) ($currencyRates[$defaultCode] ?? 1.0);
            if ($defaultRate <= 0) $defaultRate = 1.0;

            // Convert each order's amount from its stored currency to the default admin currency
            $convertAmount = function ($order) use ($currencyRates, $defaultRate) {
                $orderRate = (float) ($currencyRates[strtoupper($order->currency ?? 'USD')] ?? 1.0);
                if ($orderRate <= 0) $orderRate = 1.0;
                return ($order->amount / $orderRate) * $defaultRate;
            };

            $allCompleted = Order::completed()->get(['amount', 'currency', 'created_at']);

            $totalRevenue = $allCompleted->sum($convertAmount);
            $revenueThisMonth = $allCompleted
                ->where('created_at', '>=', now()->startOfMonth())
                ->sum($convertAmount);

            $totalOrders = Order::query()->count();
            $completedOrders = Order::completed()->count();
            $pendingOrders = Order::pending()->count();

            $revenueStats = [
                'total_revenue' => $totalRevenue,
                'revenue_this_month' => $revenueThisMonth,
                'total_orders' => $totalOrders,
                'completed_orders' => $completedOrders,
                'pending_orders' => $pendingOrders,
                'currency_symbol' => $currencyConfig['symbol'],
            ];
        }

        // Plan distribution (admin only)
        $planStats = null;
        if ($isAdmin) {
            $planOrders = Order::query()
                ->selectRaw('plan_id, COUNT(*) as order_count')
                ->groupBy('plan_id')
                ->pluck('order_count', 'plan_id')
                ->all();

            $planStats = Plan::query()
                ->where('is_active', true)
                ->get()
                ->map(function (Plan $plan) use ($planOrders) {
                    return [
                        'name' => $plan->name,
                        'orders' => $planOrders[$plan->id] ?? 0,
                        'price' => $plan->offer_price ?? $plan->price,
                    ];
                })
                ->all();
        }

        // Recent activity (admin only, last 8 entries)
        $recentActivity = null;
        if ($isAdmin) {
            $recentActivity = ActivityLog::query()
                ->with('user:id,name')
                ->latest('logged_at')
                ->limit(8)
                ->get()
                ->map(function (ActivityLog $log) {
                    return [
                        'user' => $log->user?->name ?? 'System',
                        'action' => $log->action,
                        'description' => $log->description,
                        'time' => $log->logged_at?->diffForHumans() ?? $log->created_at->diffForHumans(),
                    ];
                })
                ->all();
        }

        return view('dashboard', [
            'metrics' => [
                'active_events' => $activeEventsCount,
                'events_this_week' => $eventsThisWeek,
                'queue_count' => $queueCount,
                'storage_percent' => $storagePercent,
                'storage_used_label' => $storageSummary['used_label'],
                'storage_limit_label' => $storageSummary['limit_label'],
                'storage_warning' => $storageWarning,
                'active_users' => $activeUsersCount,
                'new_users' => $newUsersThisWeek,
                'total_media' => $totalMediaCount,
                'is_admin' => $isAdmin,
            ],
            'throughput' => $throughput,
            'eventHealth' => $eventHealth,
            'recentDeliveries' => $recentDeliveries,
            'revenueStats' => $revenueStats,
            'planStats' => $planStats,
            'recentActivity' => $recentActivity,
            'systemSnapshot' => $isAdmin ? [
                'two_factor' => $twoFactorRequired ? 'Required' : 'Optional',
                'blocked_ips' => BlockedIp::query()->count(),
                'blocked_locations' => BlockedLocation::query()->count(),
                'queue_pending' => $queueCount,
                'social_login' => count($socialProviders) > 0 ? 'Active' : 'Pending',
            ] : null,
        ]);
    }

    private function buildThroughput(int $queueCount, ?User $user = null): array
    {
        $start = now()->subDays(5)->startOfDay();
        $query = EventMedia::query()
            ->where('created_at', '>=', $start);

        if ($user) {
            $query->whereHas('event', function ($q) use ($user) {
                $q->where('created_by', $user->id);
            });
        }

        $dailyUploads = $query
            ->selectRaw('DATE(created_at) as day, COUNT(*) as total')
            ->groupBy('day')
            ->pluck('total', 'day')
            ->all();

        $values = [];
        for ($i = 5; $i >= 0; $i--) {
            $day = now()->subDays($i)->toDateString();
            $values[] = (int) ($dailyUploads[$day] ?? 0);
        }

        $count = count($values);
        $total = array_sum($values);
        $avgPerDay = $count > 0 ? (int) round($total / $count) : 0;
        $peakPerDay = $count > 0 ? (int) max($values) : 0;
        $avgPerMinute = $avgPerDay > 0 ? round($avgPerDay / 1440, 1) : 0.0;

        if ($queueCount === 0) {
            $queueEtaLabel = 'Queue is clear';
        } elseif ($avgPerMinute <= 0) {
            $queueEtaLabel = 'ETA unavailable';
        } else {
            $etaMinutes = (int) ceil($queueCount / $avgPerMinute);
            $queueEtaLabel = $this->formatDuration($etaMinutes);
        }

        return [
            'avg_per_day' => $avgPerDay,
            'avg_per_minute' => $avgPerMinute,
            'peak_per_day' => $peakPerDay,
            'queue_eta_label' => $queueEtaLabel,
            'chart' => $this->buildChart($values),
        ];
    }

    private function buildChart(array $values): array
    {
        $count = count($values);
        if ($count === 0) {
            return [
                'line' => '',
                'fill' => '',
                'points' => [],
            ];
        }

        $xStart = 20;
        $xEnd = 500;
        $yTop = 60;
        $yBottom = 160;
        $max = max($values);
        $max = $max > 0 ? $max : 1;
        $step = $count > 1 ? ($xEnd - $xStart) / ($count - 1) : 0;

        $points = [];
        foreach ($values as $index => $value) {
            $x = $xStart + ($step * $index);
            $y = $yBottom - (($value / $max) * ($yBottom - $yTop));
            $points[] = [
                'x' => round($x, 2),
                'y' => round($y, 2),
            ];
        }

        $line = 'M ' . implode(' L ', array_map(function ($point) {
            return $point['x'] . ' ' . $point['y'];
        }, $points));

        $fill = $line . ' L ' . $xEnd . ' ' . $yBottom . ' L ' . $xStart . ' ' . $yBottom . ' Z';

        return [
            'line' => $line,
            'fill' => $fill,
            'points' => $points,
        ];
    }

    private function formatBytes(int $bytes): string
    {
        if ($bytes <= 0) {
            return '0 B';
        }

        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $power = (int) floor(log($bytes, 1024));
        $power = min($power, count($units) - 1);
        $value = $bytes / (1024 ** $power);
        $precision = $power === 0 ? 0 : 1;

        return number_format($value, $precision) . ' ' . $units[$power];
    }

    private function formatDuration(int $minutes): string
    {
        if ($minutes < 60) {
            return $minutes . ' min';
        }

        $hours = (int) floor($minutes / 60);
        $remaining = $minutes % 60;

        if ($hours >= 24) {
            $days = (int) floor($hours / 24);
            $hours = $hours % 24;

            return $days . ' d ' . $hours . ' hr';
        }

        return $hours . ' hr ' . $remaining . ' min';
    }
}
