<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportEventsRequest;
use App\Http\Requests\Admin\StoreEventRequest;
use App\Http\Requests\Admin\UpdateEventRequest;
use App\Models\Event;
use App\Models\DriveImport;
use App\Services\GoogleDriveImportService;
use App\Services\PlanLimitService;
use App\Services\ExportService;
use App\Support\EventMediaSettings;
use App\Support\FormatSettings;
use App\Exceptions\PlanLimitReachedException;
use App\Exceptions\StorageLimitReachedException;
use App\Services\StorageUsageService;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Barryvdh\DomPDF\Facade\Pdf;

class EventController extends Controller
{


    public function index(Request $request)
    {
        $this->authorize('viewAny', Event::class);

        $search = $request->query('search');
        $type = $request->query('type');
        $status = $request->query('status');
        $sort = $request->query('sort', 'date_desc');

        $user = auth()->user();
        $isAdmin = $user->hasRole(['Super Admin', 'Admin']);

        $query = Event::query()->withCount('media')->with('coverMedia');

        if (! $isAdmin) {
            $query->where('created_by', $user->id);
        }

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('type', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($type) {
            $query->where('type', $type);
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return view('admin.events.index', [
            'events' => $query->paginate(10)->withQueryString(),
            'types' => config('events.types', []),
            'search' => $search,
            'typeFilter' => $type,
            'status' => $status,
            'sort' => $sort,
        ]);
    }

    public function create()
    {
        $this->authorize('create', Event::class);

        $user = auth()->user();
        $canSetWatermark = $user && ($user->hasRole(['Super Admin', 'Admin']) || $user->plan?->has_watermark);

        return view('admin.events.create', [
            'types' => config('events.types', []),
            'canSetWatermark' => $canSetWatermark,
            'watermarkPositions' => config('events.media.watermark.positions', []),
            'watermarkImage' => EventMediaSettings::getValue('watermark.image', config('events.media.watermark.image')),
            'watermarkDefaultPosition' => EventMediaSettings::getValue('watermark.position', config('events.media.watermark.position', 'top_right')),
        ]);
    }

    public function store(StoreEventRequest $request, GoogleDriveImportService $driveImporter, PlanLimitService $planLimits)
    {
        $this->authorize('create', Event::class);

        $data = $request->validated();
        $user = $request->user();

        if ($user) {
            try {
                $planLimits->assertCanCreateEvent($user, $request->boolean('is_active'));
            } catch (PlanLimitReachedException $exception) {
                return back()
                    ->withInput()
                    ->with('error', $exception->getMessage());
            }
        }

        $event = Event::create([
            'name' => $data['name'],
            'slug' => $this->uniqueSlug($data['name']),
            'type' => $data['type'],
            'details' => $data['details'] ?? null,
            'event_date' => $data['event_date'],
            'expiry_date' => $data['expiry_date'] ?? null,
            'notifications_enabled' => $request->boolean('notifications_enabled'),
            'is_active' => $request->boolean('is_active'),
            'hashtags' => $this->normalizeTags($data['hashtags'] ?? null),
            'share_token' => $this->uniqueToken(),
            'created_by' => $request->user()?->id,
            'guest_pin' => $request->filled('guest_pin') ? Crypt::encryptString($request->input('guest_pin')) : null,
            'admin_pin' => $request->filled('admin_pin') ? Crypt::encryptString($request->input('admin_pin')) : null,
            'watermark_position' => $user && ($user->hasRole(['Super Admin', 'Admin']) || $user->plan?->has_watermark) ? (($data['watermark_position'] ?? null) ?: null) : null,
        ]);

        $driveSummary = null;
        if ($request->filled('google_drive_links')) {
            try {
                $driveResult = $driveImporter->import($event, $request->input('google_drive_links'));
                $driveSummary = "Google Drive: imported {$driveResult['imported']}, skipped {$driveResult['skipped']}.";
            } catch (PlanLimitReachedException $exception) {
                return redirect()
                    ->route('admin.events.edit', $event)
                    ->with('error', $exception->getMessage());
            } catch (StorageLimitReachedException $exception) {
                return redirect()
                    ->route('admin.events.edit', $event)
                    ->with('error', $exception->getMessage());
            }
        }

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.events.create')
                ->with('status', trim('Event created. You can add another. ' . ($driveSummary ?? '')));
        }

        return redirect()
            ->route('admin.events.index')
            ->with('status', trim('Event created successfully. ' . ($driveSummary ?? '')));
    }

    public function show(Request $request, Event $event)
    {
        $this->authorize('view', $event);

        $mediaSort = $request->query('media_sort', 'newest');
        $mediaType = $request->query('media_type');

        $mediaQuery = $event->media()->orderBy('created_at', 'desc');

        if ($mediaType) {
            $mediaQuery->where('file_type', $mediaType);
        }

        $this->applyMediaSort($mediaQuery, $mediaSort);

        $chunkSize = (int) EventMediaSettings::getValue('chunk_size', config('events.media.chunk_size', 5242880));
        $ftpBasePath = (string) EventMediaSettings::getValue('ftp_path', config('events.media.ftp_path', storage_path('app/ftp')));
        $ftpBasePath = rtrim($ftpBasePath, DIRECTORY_SEPARATOR);
        $stats = $event->media()
            ->selectRaw("
                COUNT(*) as total,
                SUM(CASE WHEN file_type = 'image' THEN 1 ELSE 0 END) as images,
                SUM(CASE WHEN file_type = 'video' THEN 1 ELSE 0 END) as videos,
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                SUM(CASE WHEN status = 'processing' THEN 1 ELSE 0 END) as processing,
                SUM(CASE WHEN status = 'ready' THEN 1 ELSE 0 END) as ready,
                SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed,
                COALESCE(SUM(size), 0) as total_size
            ")
            ->first();
        $driveImports = $event->driveImports()->latest()->limit(5)->get();

        // Client selections count for tab badge
        $selectionsCount = \App\Models\ClientSelection::where('event_id', $event->id)->count();

        return view('admin.events.show', [
            'event' => $event->load('coverMedia'),
            'media' => $mediaQuery->paginate(12)->withQueryString(),
            'mediaSort' => $mediaSort,
            'mediaType' => $mediaType,
            'chunkSize' => $chunkSize,
            'ftpBasePath' => $ftpBasePath,
            'driveImports' => $driveImports,
            'mediaCounts' => [
                'total' => (int) $stats->total,
                'images' => (int) $stats->images,
                'videos' => (int) $stats->videos,
            ],
            'mediaPipeline' => [
                'pending' => (int) $stats->pending,
                'processing' => (int) $stats->processing,
                'ready' => (int) $stats->ready,
                'failed' => (int) $stats->failed,
                'size_label' => StorageUsageService::formatBytes((int) $stats->total_size),
                'queue_total' => (int) $stats->pending + (int) $stats->processing,
            ],
            'watermarkEnabled' => filter_var(
                EventMediaSettings::getValue('watermark.enabled', config('events.media.watermark.enabled', true)),
                FILTER_VALIDATE_BOOLEAN
            ),
            'selectionsCount' => $selectionsCount,
        ]);
    }

    public function invoice(Event $event)
    {
        $this->authorize('view', $event);

        $event->load('createdBy');
        $totalMedia = $event->media()->count();
        $imageCount = $event->media()->where('file_type', 'image')->count();
        $videoCount = $event->media()->where('file_type', 'video')->count();
        $mediaBytes = (int) $event->media()->sum('size');

        $invoiceNumber = 'INV-EVENT-' . $event->id . '-' . now()->format('Ymd');
        $issuedAt = now();

        $pdf = Pdf::loadView('admin.events.invoice', [
            'event' => $event,
            'invoiceNumber' => $invoiceNumber,
            'issuedAt' => $issuedAt,
            'mediaCounts' => [
                'total' => $totalMedia,
                'images' => $imageCount,
                'videos' => $videoCount,
            ],
            'storageUsed' => StorageUsageService::formatBytes($mediaBytes),
        ])->setPaper('a4', 'portrait');

        $filename = 'event-' . $event->id . '-invoice-' . $issuedAt->format('Ymd-His') . '.pdf';

        return $pdf->download($filename);
    }

    public function edit(Event $event)
    {
        $this->authorize('update', $event);

        if ($event->guest_pin) {
            try {
                $event->guest_pin = Crypt::decryptString($event->guest_pin);
            } catch (\Exception $e) {
                $event->guest_pin = null;
            }
        }

        if ($event->admin_pin) {
            try {
                $event->admin_pin = Crypt::decryptString($event->admin_pin);
            } catch (\Exception $e) {
                $event->admin_pin = null;
            }
        }

        $user = auth()->user();
        $canSetWatermark = $user && ($user->hasRole(['Super Admin', 'Admin']) || $user->plan?->has_watermark);

        return view('admin.events.edit', [
            'event' => $event,
            'types' => config('events.types', []),
            'canSetWatermark' => $canSetWatermark,
            'watermarkPositions' => config('events.media.watermark.positions', []),
            'watermarkImage' => EventMediaSettings::getValue('watermark.image', config('events.media.watermark.image')),
            'watermarkDefaultPosition' => EventMediaSettings::getValue('watermark.position', config('events.media.watermark.position', 'top_right')),
        ]);
    }

    public function update(UpdateEventRequest $request, Event $event, GoogleDriveImportService $driveImporter, PlanLimitService $planLimits)
    {
        $this->authorize('update', $event);

        $data = $request->validated();
        $user = $request->user();

        if ($user && ! $event->is_active && $request->boolean('is_active')) {
            try {
                $planLimits->assertCanCreateEvent($user, true);
            } catch (PlanLimitReachedException $exception) {
                return back()
                    ->withInput()
                    ->with('error', $exception->getMessage());
            }
        }

        $payload = [
            'name' => $data['name'],
            'type' => $data['type'],
            'details' => $data['details'] ?? null,
            'event_date' => $data['event_date'],
            'expiry_date' => $data['expiry_date'] ?? null,
            'notifications_enabled' => $request->boolean('notifications_enabled'),
            'allow_guest_upload' => $request->boolean('allow_guest_upload'),
            'allow_public_downloads' => $request->boolean('allow_public_downloads'),
            'is_active' => $request->boolean('is_active'),
            'hashtags' => $this->normalizeTags($data['hashtags'] ?? null),
        ];

        if ($user && ($user->hasRole(['Super Admin', 'Admin']) || $user->plan?->has_watermark)) {
            $payload['watermark_position'] = ($data['watermark_position'] ?? null) ?: null;
        }

        if ($data['name'] !== $event->name) {
            $payload['slug'] = $this->uniqueSlug($data['name'], $event->id);
        }

        if ($request->filled('guest_pin')) {
            $payload['guest_pin'] = Crypt::encryptString($request->input('guest_pin'));
        }

        if ($request->filled('admin_pin')) {
            $payload['admin_pin'] = Crypt::encryptString($request->input('admin_pin'));
        }

        $event->update($payload);

        $driveSummary = null;
        if ($request->filled('google_drive_links')) {
            try {
                $driveResult = $driveImporter->import($event, $request->input('google_drive_links'));
                $driveSummary = "Google Drive: imported {$driveResult['imported']}, skipped {$driveResult['skipped']}.";
            } catch (PlanLimitReachedException $exception) {
                return back()
                    ->with('error', $exception->getMessage());
            } catch (StorageLimitReachedException $exception) {
                return back()
                    ->with('error', $exception->getMessage());
            }
        }

        return back()
            ->with('status', trim('Event updated successfully. ' . ($driveSummary ?? '')));
    }

    public function destroy(Event $event)
    {
        $this->authorize('delete', $event);

        $mediaCount = $event->media()->count();

        // Permanently delete all media files from disk, then delete the event
        DB::transaction(function () use ($event) {
            // forceDelete() triggers the forceDeleting hook which removes physical files
            $event->media()->withTrashed()->chunkById(100, function ($media) {
                foreach ($media as $m) {
                    $m->forceDelete();
                }
            });
            $event->delete();
        });

        return redirect()
            ->route('admin.events.index')
            ->with('status', "Event and {$mediaCount} media file(s) deleted permanently.");
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', Event::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:events,id'],
        ]);

        $totalMedia = 0;

        // Permanently delete all media files from disk, then delete events
        DB::transaction(function () use ($validated, &$totalMedia) {
            $events = Event::query()->whereIn('id', $validated['ids'])->get();

            foreach ($events as $event) {
                $totalMedia += $event->media()->count();
                // forceDelete() triggers the forceDeleting hook which removes physical files
                $event->media()->withTrashed()->chunkById(100, function ($media) {
                    foreach ($media as $m) {
                        $m->forceDelete();
                    }
                });
                $event->delete();
            }
        });

        return redirect()
            ->route('admin.events.index')
            ->with('status', count($validated['ids']) . " event(s) and {$totalMedia} media file(s) deleted permanently.");
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', Event::class);

        $search = $request->query('search');
        $type = $request->query('type');
        $status = $request->query('status');
        $sort = $request->query('sort', 'date_desc');
        $format = (string) $request->query('format', 'csv');

        $query = Event::query()->withCount('media');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('type', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($type) {
            $query->where('type', $type);
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'name', 'label' => 'Name'],
            ['key' => 'slug', 'label' => 'Slug'],
            ['key' => 'type', 'label' => 'Type'],
            ['key' => 'event_date', 'label' => 'Event Date'],
            ['key' => 'expiry_date', 'label' => 'Expiry Date'],
            ['key' => 'notifications_enabled', 'label' => 'Notifications Enabled'],
            ['key' => 'is_active', 'label' => 'Status'],
            ['key' => 'hashtags', 'label' => 'Hashtags'],
            ['key' => 'media_count', 'label' => 'Media Count'],
        ];

        $map = function (Event $event) {
            return [
                'name' => $event->name,
                'slug' => $event->slug,
                'type' => $event->type,
                'event_date' => $event->event_date ? FormatSettings::date($event->event_date) : '',
                'expiry_date' => $event->expiry_date ? FormatSettings::date($event->expiry_date) : '',
                'notifications_enabled' => $event->notifications_enabled ? 'Yes' : 'No',
                'is_active' => $event->is_active ? 'Active' : 'Inactive',
                'hashtags' => $event->hashtags ? implode('|', $event->hashtags) : '',
                'media_count' => $event->media_count,
            ];
        };

        $filename = 'events-' . now()->format('Ymd-His');

        return $exporter->download(
            $format,
            $filename,
            $query,
            $columns,
            $map,
            'Events',
            'Events export'
        );
    }

    public function import(ImportEventsRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.events.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.events.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, [
            'name',
            'type',
            'event_date',
            'expiry_date',
            'notifications_enabled',
            'is_active',
            'hashtags',
        ]);

        $imported = 0;
        $skipped = 0;
        $maxRows = 10000;
        $rowCount = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (++$rowCount > $maxRows) {
                break;
            }
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ''));
            $type = $this->normalizeType($data['type'] ?? null);
            $eventDate = $this->parseDate($data['event_date'] ?? null);

            if ($name === '' || ! $type || ! $eventDate) {
                $skipped++;
                continue;
            }

            $expiryDate = $this->parseDate($data['expiry_date'] ?? null);
            $hashtags = $this->normalizeTags($data['hashtags'] ?? null);

            $event = Event::query()
                ->where('name', $name)
                ->whereDate('event_date', $eventDate)
                ->first();

            if ($event) {
                $event->update([
                    'type' => $type,
                    'expiry_date' => $expiryDate,
                    'notifications_enabled' => $this->parseBool($data['notifications_enabled'] ?? null, true),
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                    'hashtags' => $hashtags,
                ]);
            } else {
                Event::create([
                    'name' => $name,
                    'slug' => $this->uniqueSlug($name),
                    'type' => $type,
                    'event_date' => $eventDate,
                    'expiry_date' => $expiryDate,
                    'notifications_enabled' => $this->parseBool($data['notifications_enabled'] ?? null, true),
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                    'hashtags' => $hashtags,
                    'share_token' => $this->uniqueToken(),
                ]);
            }

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.events.index')
            ->with('status', "Imported {$imported} events. Skipped {$skipped} rows.");
    }

    private function normalizeTags(?string $value): ?array
    {
        if (! $value) {
            return null;
        }

        $tags = collect(preg_split('/[|,]/', $value))
            ->map(fn ($tag) => trim($tag))
            ->filter()
            ->unique()
            ->values()
            ->all();

        return $tags ?: null;
    }

    private function uniqueSlug(string $name, ?int $ignoreId = null): string
    {
        $base = Str::slug($name);
        $slug = $base !== '' ? $base : Str::random(8);
        $counter = 1;

        while ($this->slugExists($slug, $ignoreId)) {
            $slug = $base . '-' . $counter;
            $counter++;
        }

        return $slug;
    }

    private function slugExists(string $slug, ?int $ignoreId): bool
    {
        $query = Event::query()->where('slug', $slug);
        if ($ignoreId) {
            $query->where('id', '!=', $ignoreId);
        }

        return $query->exists();
    }

    private function uniqueToken(): string
    {
        do {
            $token = Str::random(32);
        } while (Event::query()->where('share_token', $token)->exists());

        return $token;
    }

    private function normalizeType(?string $value): ?string
    {
        if (! $value) {
            return null;
        }

        $types = config('events.types', []);
        foreach ($types as $type) {
            if (Str::lower($type) === Str::lower(trim($value))) {
                return $type;
            }
        }

        return null;
    }

    private function parseDate(?string $value): ?string
    {
        if (! $value) {
            return null;
        }

        try {
            return Carbon::parse($value)->format('Y-m-d');
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_asc' => $query->orderBy('name'),
            'name_desc' => $query->orderBy('name', 'desc'),
            'date_asc' => $query->orderBy('event_date'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('event_date', 'desc'),
        };
    }

    private function applyMediaSort($query, string $sort): void
    {
        match ($sort) {
            'oldest' => $query->orderBy('created_at', 'asc'),
            'size_asc' => $query->orderBy('size'),
            'size_desc' => $query->orderBy('size', 'desc'),
            'name_asc' => $query->orderBy('file_name'),
            'name_desc' => $query->orderBy('file_name', 'desc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }

    public function design(Event $event)
    {
        $this->authorize('accessDesignEditor', $event);

        return view('admin.events.design', compact('event'));
    }

    public function updateDesign(Request $request, Event $event)
    {
        $this->authorize('accessDesignEditor', $event);

        $allowedKeys = [
            'font_family', 'font_size',
            'grid_gap', 'image_radius', 'grid_layout', 'hero_style',
            'card_style', 'pattern',
            'primary_color', 'background_color', 'navbar_bg_color', 'navbar_text_color',
            'surface_color', 'text_color',
            'hero_gradient', 'card_shadow', 'card_radius', 'button_radius',
            'template_key', 'template_name', 'template_category', 'template_style',
            'tab_style',
        ];

        $validated = $request->validate([
            'design_settings' => ['nullable', 'array'],
            'design_settings.font_family' => ['nullable', 'string', 'max:100'],
            'design_settings.font_size' => ['nullable', 'integer', 'min:12', 'max:80'],
            'design_settings.grid_gap' => ['nullable', 'integer', 'min:0', 'max:100'],
            'design_settings.image_radius' => ['nullable', 'integer', 'min:0', 'max:100'],
            'design_settings.grid_layout' => ['nullable', 'string', 'in:masonry,square,collage'],
            'design_settings.hero_style' => ['nullable', 'string', 'in:standard,split,minimal'],
            'design_settings.card_style' => ['nullable', 'string', 'in:default,elevated,glass,minimal,bordered'],
            'design_settings.pattern' => ['nullable', 'string', 'in:none,dots,lines'],
            'design_settings.primary_color' => ['nullable', 'string', 'max:50'],
            'design_settings.background_color' => ['nullable', 'string', 'max:50'],
            'design_settings.navbar_bg_color' => ['nullable', 'string', 'max:50'],
            'design_settings.navbar_text_color' => ['nullable', 'string', 'max:50'],
            'design_settings.surface_color' => ['nullable', 'string', 'max:50'],
            'design_settings.text_color' => ['nullable', 'string', 'max:50'],
            'design_settings.hero_gradient' => ['nullable', 'string', 'max:500'],
            'design_settings.card_shadow' => ['nullable', 'string', 'max:200'],
            'design_settings.card_radius' => ['nullable', 'integer', 'min:0', 'max:100'],
            'design_settings.button_radius' => ['nullable', 'integer', 'min:0', 'max:100'],
            'design_settings.template_key' => ['nullable', 'string', 'max:50'],
            'design_settings.template_name' => ['nullable', 'string', 'max:100'],
            'design_settings.template_category' => ['nullable', 'string', 'max:50'],
            'design_settings.template_style' => ['nullable', 'string', 'max:50'],
            'design_settings.tab_style' => ['nullable', 'string', 'max:30'],
        ]);

        $settings = $validated['design_settings'] ?? [];
        $filtered = array_intersect_key($settings, array_flip($allowedKeys));

        $event->update(['design_settings' => $filtered]);

        return response()->json(['message' => 'Design settings saved.']);
    }
}
