<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Exceptions\PlanLimitReachedException;
use App\Exceptions\StorageLimitReachedException;
use App\Http\Requests\Admin\ChunkEventMediaRequest;
use App\Http\Requests\Admin\StoreEventMediaRequest;
use App\Models\Event;
use App\Models\EventMedia;
use App\Models\DriveImport;
use App\Services\StorageUsageService;
use App\Services\EventMediaService;
use App\Services\ExportService;
use App\Services\GoogleDriveImportService;
use App\Jobs\ProcessGoogleDriveImport;
use App\Support\EventMediaSettings;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use ZipArchive;

class EventMediaController extends Controller
{
    /**
     * Process pending queue jobs via AJAX (no cron/terminal needed).
     * Called automatically by frontend JS when pending media exists.
     */
    public function processQueue(Request $request)
    {
        $jobsPending = DB::table('jobs')->count();
        $mediaPending = EventMedia::whereIn('status', ['pending', 'processing'])->count();

        // Nothing left in queue AND no pending media — truly done
        if ($jobsPending === 0 && $mediaPending === 0) {
            return response()->json(['processed' => 0, 'remaining' => 0]);
        }

        if ($jobsPending > 0) {
            // Process a small batch per call so the frontend progress bar moves visibly
            Artisan::call('queue:work', [
                '--stop-when-empty' => true,
                '--max-jobs' => 3,
                '--max-time' => 30,
                '--tries' => 3,
                '--memory' => 256,
                '--quiet' => true,
            ]);
        }

        $jobsRemaining = DB::table('jobs')->count();
        $mediaStillPending = EventMedia::whereIn('status', ['pending', 'processing'])->count();
        $remaining = max($jobsRemaining, $mediaStillPending);

        return response()->json([
            'processed' => max(0, ($jobsPending + $mediaPending) - ($jobsRemaining + $mediaStillPending)),
            'remaining' => $remaining,
        ]);
    }

    public function store(StoreEventMediaRequest $request, Event $event, EventMediaService $service)
    {
        $this->authorize('update', $event);
        set_time_limit(300); // 5 minutes for batch uploads

        $files = $request->file('media', []);

        try {
            $uploaded = $service->storeBatch($event, $files);
        } catch (PlanLimitReachedException $exception) {
            $message = $this->planLimitMessage($exception);

            if ($request->expectsJson()) {
                return response()->json([
                    'error' => $message,
                ], 422);
            }

            return redirect()
                ->route('admin.events.show', $event)
                ->with('error', $message);
        } catch (StorageLimitReachedException $exception) {
            $message = $this->storageLimitMessage($exception);

            if ($request->expectsJson()) {
                return response()->json([
                    'error' => $message,
                ], 422);
            }

            return redirect()
                ->route('admin.events.show', $event)
                ->with('error', $message);
        }

        if ($request->expectsJson()) {
            return response()->json([
                'status' => "Uploaded {$uploaded} file(s). Processing now...",
            ]);
        }

        return redirect()
            ->route('admin.events.show', $event)
            ->with('status', "Uploaded {$uploaded} file(s). Processing now...");
    }

    public function storeZip(Request $request, Event $event, EventMediaService $service)
    {
        $this->authorize('update', $event);
        set_time_limit(600); // 10 minutes for ZIP extraction + storage

        $request->validate([
            'zip' => ['required', 'file', 'mimes:zip', 'max:512000'], // 500MB max
        ]);

        try {
            $uploaded = $service->storeZip($event, $request->file('zip'));
        } catch (PlanLimitReachedException $exception) {
            $message = $this->planLimitMessage($exception);

            if ($request->expectsJson()) {
                return response()->json(['error' => $message], 422);
            }

            return redirect()->route('admin.events.show', $event)->with('error', $message);
        } catch (StorageLimitReachedException $exception) {
            $message = $this->storageLimitMessage($exception);

            if ($request->expectsJson()) {
                return response()->json(['error' => $message], 422);
            }

            return redirect()->route('admin.events.show', $event)->with('error', $message);
        } catch (\RuntimeException $e) {
            if ($request->expectsJson()) {
                return response()->json(['error' => $e->getMessage()], 422);
            }

            return redirect()->route('admin.events.show', $event)->with('error', $e->getMessage());
        }

        if ($request->expectsJson()) {
            return response()->json([
                'status' => "Extracted {$uploaded} file(s) from ZIP. Processing now...",
                'count' => $uploaded,
            ]);
        }

        return redirect()
            ->route('admin.events.show', $event)
            ->with('status', "Extracted {$uploaded} file(s) from ZIP. Processing now...");
    }

    public function storeChunk(ChunkEventMediaRequest $request, Event $event, EventMediaService $service)
    {
        $uploadId = $request->input('upload_id');
        $chunkIndex = (int) $request->input('chunk_index');
        $totalChunks = (int) $request->input('total_chunks');
        $fileName = $request->input('file_name');
        $chunkFile = $request->file('chunk');
        $expectedChecksum = $request->input('chunk_checksum');

        // SECURITY FIX #2: Validate chunk integrity with checksum
        if ($expectedChecksum) {
            $actualChecksum = hash_file('sha256', $chunkFile->getRealPath());
            if ($actualChecksum !== $expectedChecksum) {
                \Log::warning('Chunk checksum mismatch', [
                    'upload_id' => $uploadId,
                    'chunk_index' => $chunkIndex,
                    'expected' => $expectedChecksum,
                    'actual' => $actualChecksum,
                ]);

                return response()->json([
                    'error' => 'Chunk integrity check failed. Please retry upload.',
                ], 422);
            }
        }

        $chunkDirectory = 'tmp/event-chunks/' . $uploadId;
        Storage::disk('local')->makeDirectory($chunkDirectory);
        Storage::disk('local')->putFileAs($chunkDirectory, $chunkFile, 'chunk_' . $chunkIndex);

        if ($chunkIndex + 1 < $totalChunks) {
            return response()->json([
                'status' => 'chunk_received',
                'received' => $chunkIndex + 1,
                'total' => $totalChunks,
            ]);
        }

        // Prevent PHP timeout during assembly + cloud upload
        set_time_limit(300);

        // RELIABILITY FIX #2: File locking and guaranteed cleanup
        $localDisk = Storage::disk('local');
        $assembledPath = $localDisk->path($chunkDirectory . '/assembled_' . Str::random(8));
        $chunkDirectoryFull = $localDisk->path($chunkDirectory);

        try {
            $output = fopen($assembledPath, 'wb');
            if (! $output) {
                throw new \RuntimeException('Failed to create assembled file');
            }

            // CRITICAL: Lock file during assembly to prevent race conditions
            if (! flock($output, LOCK_EX)) {
                fclose($output);
                throw new \RuntimeException('Could not acquire file lock for assembly');
            }

            try {
                // Assemble chunks with validation
                for ($i = 0; $i < $totalChunks; $i++) {
                    $chunkPath = Storage::disk('local')->path($chunkDirectory . '/chunk_' . $i);

                    if (! file_exists($chunkPath)) {
                        throw new \RuntimeException("Chunk {$i} not found");
                    }

                    $input = fopen($chunkPath, 'rb');
                    if (! $input) {
                        throw new \RuntimeException("Cannot read chunk {$i}");
                    }

                    stream_copy_to_stream($input, $output);
                    fclose($input);
                }

                // Release lock and close file
                flock($output, LOCK_UN);
                fclose($output);

                // Process assembled file
                $media = $service->storeLocalFile($event, $assembledPath, $fileName);

                return response()->json([
                    'status' => 'completed',
                    'media_id' => $media?->id,
                ]);

            } catch (\Throwable $assemblyError) {
                // Release lock on error
                if (is_resource($output)) {
                    flock($output, LOCK_UN);
                    fclose($output);
                }
                throw $assemblyError;
            }
        } catch (PlanLimitReachedException $exception) {
            return response()->json([
                'error' => $this->planLimitMessage($exception),
            ], 422);
        } catch (StorageLimitReachedException $exception) {
            return response()->json([
                'error' => $this->storageLimitMessage($exception),
            ], 422);
        } catch (\Throwable $e) {
            \Log::error('Chunk assembly failed', [
                'upload_id' => $uploadId,
                'event_id' => $event->id,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'error' => 'Upload assembly failed. Please try again.',
            ], 500);
        } finally {
            // CRITICAL: Always cleanup temp files
            Storage::disk('local')->deleteDirectory($chunkDirectory);
            if (isset($assembledPath) && is_file($assembledPath)) {
                @unlink($assembledPath);
            }
        }
    }

    public function destroy(Event $event, EventMedia $media)
    {
        $this->ensureBelongsToEvent($event, $media);

        // forceDelete() removes files from disk via the forceDeleting model hook
        $media->forceDelete();

        return redirect()
            ->route('admin.events.show', $event)
            ->with('status', 'Media deleted permanently.');
    }

    public function bulkDestroy(Request $request, Event $event)
    {
        $this->authorize('update', $event);

        // Select-all-pages: delete ALL media for this event
        if ($request->boolean('select_all_pages')) {
            $count = 0;
            EventMedia::query()
                ->where('event_id', $event->id)
                ->chunkById(100, function ($chunk) use (&$count) {
                    foreach ($chunk as $media) {
                        $media->forceDelete();
                        $count++;
                    }
                });

            return redirect()
                ->route('admin.events.show', $event)
                ->with('status', $count . ' media file(s) deleted permanently.');
        }

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:event_media,id'],
        ]);

        // Permanently delete media and their files from disk
        DB::transaction(function () use ($event, $validated) {
            $mediaItems = EventMedia::query()
                ->where('event_id', $event->id)
                ->whereIn('id', $validated['ids'])
                ->get();

            foreach ($mediaItems as $media) {
                $media->forceDelete();
            }
        });

        return redirect()
            ->route('admin.events.show', $event)
            ->with('status', count($validated['ids']) . ' media file(s) deleted permanently.');
    }

    public function setCover(Event $event, EventMedia $media)
    {
        $this->authorize('update', $event);
        $this->ensureBelongsToEvent($event, $media);

        if ($media->file_type !== 'image') {
            return redirect()
                ->route('admin.events.show', $event)
                ->with('error', 'Only images can be set as the cover.');
        }

        $event->media()->where('is_cover', true)->update(['is_cover' => false]);
        $media->update(['is_cover' => true]);

        return redirect()
            ->route('admin.events.show', $event)
            ->with('status', 'Cover image updated successfully.');
    }

    public function toggleFeatured(Event $event, EventMedia $media)
    {
        $this->authorize('update', $event);
        $this->ensureBelongsToEvent($event, $media);

        $media->update(['is_featured' => ! $media->is_featured]);

        return redirect()
            ->route('admin.events.show', $event)
            ->with('status', 'Featured status updated.');
    }

    public function download(Request $request, Event $event, EventMedia $media)
    {
        $this->ensureBelongsToEvent($event, $media);

        $type = $request->query('type', 'optimized');
        if ($type === 'original') {
            return Storage::disk($media->originalDisk())->download($media->original_path, $media->file_name);
        }

        $diskName = $media->disk;
        $path = $media->optimized_path;

        if (! $path || ! Storage::disk($diskName)->exists($path)) {
            $diskName = $media->originalDisk();
            $path = $media->original_path;
        }

        return Storage::disk($diskName)->download($path, $media->file_name);
    }

    public function bulkDownload(Request $request, Event $event)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:event_media,id'],
        ]);

        $mediaItems = EventMedia::query()
            ->where('event_id', $event->id)
            ->whereIn('id', $validated['ids'])
            ->get();

        if ($mediaItems->isEmpty()) {
            return redirect()
                ->route('admin.events.show', $event)
                ->with('error', 'No media selected for download.');
        }

        $tempDir = storage_path('app/tmp/bulk-downloads');
        File::ensureDirectoryExists($tempDir);

        $zipName = 'event-' . $event->id . '-media-' . now()->format('Ymd-His') . '-' . Str::random(6) . '.zip';
        $zipPath = $tempDir . DIRECTORY_SEPARATOR . $zipName;

        $zip = new ZipArchive();
        if ($zip->open($zipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            return redirect()
                ->route('admin.events.show', $event)
                ->with('error', 'Unable to prepare the download archive.');
        }

        $usedNames = [];
        $tempFiles = [];

        foreach ($mediaItems as $media) {
            $diskName = $media->disk;
            $path = $media->optimized_path;

            if (! $path || ! Storage::disk($diskName)->exists($path)) {
                $diskName = $media->originalDisk();
                $path = $media->original_path;
            }

            if (! $path || ! Storage::disk($diskName)->exists($path)) {
                continue;
            }

            $fileName = $media->file_name ?: ('media-' . $media->id);
            $fileName = $this->uniqueZipName($fileName, $usedNames);

            $stream = Storage::disk($diskName)->readStream($path);
            if (! $stream) {
                continue;
            }

            $tempFile = $tempDir . DIRECTORY_SEPARATOR . 'tmp-' . Str::random(10);
            $output = fopen($tempFile, 'wb');
            if (! $output) {
                fclose($stream);
                continue;
            }
            stream_copy_to_stream($stream, $output);
            fclose($stream);
            fclose($output);

            $zip->addFile($tempFile, $fileName);
            $tempFiles[] = $tempFile;
        }

        $zip->close();

        foreach ($tempFiles as $tempFile) {
            if (is_file($tempFile)) {
                @unlink($tempFile);
            }
        }

        if (! file_exists($zipPath)) {
            return redirect()
                ->route('admin.events.show', $event)
                ->with('error', 'Unable to create the download archive.');
        }

        return response()->download($zipPath, $zipName)->deleteFileAfterSend(true);
    }

    public function export(Request $request, Event $event, ExportService $exporter)
    {
        $mediaType = $request->query('media_type');
        $mediaSort = $request->query('media_sort', 'newest');
        $format = (string) $request->query('format', 'csv');

        $query = $event->media()->newQuery();

        if ($mediaType) {
            $query->where('file_type', $mediaType);
        }

        $this->applyMediaSort($query, $mediaSort);

        $columns = [
            ['key' => 'file_name', 'label' => 'File Name'],
            ['key' => 'file_type', 'label' => 'Type'],
            ['key' => 'size', 'label' => 'Size'],
            ['key' => 'resolution', 'label' => 'Resolution'],
            ['key' => 'status', 'label' => 'Status'],
            ['key' => 'featured', 'label' => 'Featured'],
            ['key' => 'cover', 'label' => 'Cover'],
            ['key' => 'guest_upload', 'label' => 'Guest Upload'],
            ['key' => 'uploaded_at', 'label' => 'Uploaded At'],
        ];

        $map = function (EventMedia $media) {
            $sizeLabel = $media->size ? number_format($media->size / 1024, 1) . ' KB' : 'N/A';
            $resolution = $media->width && $media->height ? $media->width . 'x' . $media->height : 'N/A';

            return [
                'file_name' => $media->file_name,
                'file_type' => strtoupper((string) $media->file_type),
                'size' => $sizeLabel,
                'resolution' => $resolution,
                'status' => ucfirst((string) $media->status),
                'featured' => $media->is_featured ? 'Yes' : 'No',
                'cover' => $media->is_cover ? 'Yes' : 'No',
                'guest_upload' => $media->is_guest_upload ? 'Yes' : 'No',
                'uploaded_at' => optional($media->created_at)->format('Y-m-d H:i:s'),
            ];
        };

        $filename = 'event-' . $event->id . '-media-' . now()->format('Ymd-His');

        return $exporter->download(
            $format,
            $filename,
            $query,
            $columns,
            $map,
            'Event Media Export',
            'Media export for event "' . $event->name . '".'
        );
    }

    public function ingestFtp(Event $event, EventMediaService $service)
    {
        $this->authorize('update', $event);

        // Enforce plan feature: FTP import (Admin/Super Admin bypass automatically)
        $planLimit = app(\App\Services\PlanLimitService::class);
        if (! $planLimit->hasFeature(auth()->user(), 'has_ftp_import')) {
            return redirect()
                ->route('admin.events.show', $event)
                ->with('error', 'FTP tethering is not available on your current plan. Please upgrade.');
        }

        $basePath = (string) EventMediaSettings::getValue('ftp_path', config('events.media.ftp_path'));
        $basePath = rtrim($basePath, DIRECTORY_SEPARATOR);
        $path = $basePath . DIRECTORY_SEPARATOR . $event->id;

        if (! File::isDirectory($path)) {
            return redirect()
                ->route('admin.events.show', $event)
                ->with('error', 'FTP base directory is not configured or not found.');
        }

        $processedDir = $path . DIRECTORY_SEPARATOR . 'processed';
        File::ensureDirectoryExists($processedDir);

        $imported = 0;
        $allowed = ['jpg', 'jpeg', 'png', 'webp', 'mp4'];

        foreach (File::files($path) as $file) {
            if (! $file->isFile()) {
                continue;
            }

            $extension = strtolower($file->getExtension());
            if (! in_array($extension, $allowed, true)) {
                continue;
            }

            try {
                $media = $service->storeLocalFile($event, $file->getPathname(), $file->getFilename());
            } catch (PlanLimitReachedException $exception) {
                return redirect()
                    ->route('admin.events.show', $event)
                    ->with('error', $this->planLimitMessage($exception));
            } catch (StorageLimitReachedException $exception) {
                return redirect()
                    ->route('admin.events.show', $event)
                    ->with('error', $this->storageLimitMessage($exception));
            }
            if ($media) {
                $imported++;
                File::move($file->getPathname(), $processedDir . DIRECTORY_SEPARATOR . $file->getFilename());
            }
        }

        return redirect()
            ->route('admin.events.show', $event)
            ->with('status', "Imported {$imported} file(s) from FTP tethering.");
    }

    public function importGoogleDrive(Request $request, Event $event, GoogleDriveImportService $driveImporter)
    {
        $this->authorize('update', $event);

        // Enforce plan feature: Google Drive import (Admin/Super Admin bypass automatically)
        $planLimit = app(\App\Services\PlanLimitService::class);
        if (! $planLimit->hasFeature(auth()->user(), 'has_google_drive_import')) {
            return redirect()
                ->route('admin.events.show', $event)
                ->with('error', 'Google Drive import is not available on your current plan. Please upgrade.');
        }

        $validated = $request->validate([
            'google_drive_link' => ['required', 'string', 'max:2000'],
        ]);

        $link = trim($validated['google_drive_link']);

        try {
            if (! $driveImporter->looksLikeFolderLink($link)) {
                throw new \InvalidArgumentException('Please enter a public Google Drive folder link.');
            }

            $session = $driveImporter->startImportSession($event, $link);
            ProcessGoogleDriveImport::dispatch($event->id);
        } catch (\InvalidArgumentException $exception) {
            return redirect()
                ->route('admin.events.show', $event)
                ->withErrors(['google_drive_link' => $exception->getMessage()])
                ->withInput()
                ->with('open_google_drive_modal', true);
        } catch (\Throwable $exception) {
            return redirect()
                ->route('admin.events.show', $event)
                ->withErrors(['google_drive_link' => 'Unable to start Google Drive import: ' . $exception->getMessage()])
                ->withInput()
                ->with('open_google_drive_modal', true);
        }

        return redirect()
            ->route('admin.events.show', $event)
            ->with('status', 'Google Drive import started. Progress will update automatically.');
    }

    public function googleDriveStatus(Request $request, Event $event, GoogleDriveImportService $driveImporter)
    {
        $driveImporter->runForEvent($event, 3, 5);

        $imports = DriveImport::query()
            ->where('event_id', $event->id)
            ->orderByDesc('created_at')
            ->limit(5)
            ->get(['id', 'status', 'total_files', 'processed_files', 'failed_files', 'discovered_files', 'last_error', 'created_at']);

        return response()->json([
            'imports' => $imports->map(function (DriveImport $import) {
                $remaining = max(0, ($import->total_files ?? 0) - ($import->processed_files ?? 0) - ($import->failed_files ?? 0));

                return [
                    'id' => $import->id,
                    'status' => $import->status,
                    'total_files' => $import->total_files,
                    'discovered_files' => $import->discovered_files,
                    'processed_files' => $import->processed_files,
                    'failed_files' => $import->failed_files,
                    'remaining' => $remaining,
                    'last_error' => $import->last_error,
                    'created_at' => optional($import->created_at)->toDateTimeString(),
                ];
            }),
        ]);
    }

    private function ensureBelongsToEvent(Event $event, EventMedia $media): void
    {
        if ($media->event_id !== $event->id) {
            abort(404);
        }
    }

    private function uniqueZipName(string $fileName, array &$usedNames): string
    {
        $name = $fileName;
        $counter = 1;

        while (in_array($name, $usedNames, true)) {
            $extension = pathinfo($fileName, PATHINFO_EXTENSION);
            $basename = pathinfo($fileName, PATHINFO_FILENAME);
            $suffix = '-' . $counter;
            $name = $extension !== '' ? $basename . $suffix . '.' . $extension : $basename . $suffix;
            $counter++;
        }

        $usedNames[] = $name;

        return $name;
    }

    private function applyMediaSort($query, string $sort): void
    {
        match ($sort) {
            'oldest' => $query->orderBy('created_at', 'asc'),
            'size_asc' => $query->orderBy('size'),
            'size_desc' => $query->orderBy('size', 'desc'),
            'name_asc' => $query->orderBy('file_name'),
            'name_desc' => $query->orderBy('file_name', 'desc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }

    private function storageLimitMessage(StorageLimitReachedException $exception): string
    {
        $usedLabel = StorageUsageService::formatBytes($exception->usedBytes());
        $limitLabel = StorageUsageService::formatBytes($exception->limitBytes());

        return 'Storage limit reached. Used ' . $usedLabel . ' of ' . $limitLabel . '.';
    }

    private function planLimitMessage(PlanLimitReachedException $exception): string
    {
        return match ($exception->limitType) {
            'storage' => 'Plan storage limit reached. Used ' . StorageUsageService::formatBytes($exception->used) . ' of ' . StorageUsageService::formatBytes($exception->limit) . '.',
            'images' => 'Plan image limit reached. You have ' . $exception->used . ' of ' . $exception->limit . ' images.',
            'events' => 'Plan event limit reached. You have ' . $exception->used . ' of ' . $exception->limit . ' active events.',
            default => $exception->getMessage(),
        };
    }

}
