<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportFactStatsRequest;
use App\Http\Requests\Admin\StoreFactStatRequest;
use App\Http\Requests\Admin\UpdateFactStatRequest;
use App\Models\FactStat;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class FactStatController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = FactStat::query()->with('creator');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('fact', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('value', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return view('admin.content.fact-stats.index', [
            'facts' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'status' => $status,
            'sort' => $sort,
        ]);
    }

    public function create()
    {
        return view('admin.content.fact-stats.create');
    }

    public function store(StoreFactStatRequest $request)
    {
        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');
        $data['created_by'] = $request->user()?->id;

        FactStat::create($data);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.fact-stats.create')
                ->with('status', 'Fact created. You can add another.');
        }

        return redirect()
            ->route('admin.fact-stats.index')
            ->with('status', 'Fact created successfully.');
    }

    public function show(FactStat $factStat)
    {
        $factStat->load('creator');

        return view('admin.content.fact-stats.show', [
            'fact' => $factStat,
        ]);
    }

    public function edit(FactStat $factStat)
    {
        $factStat->load('creator');

        return view('admin.content.fact-stats.edit', [
            'fact' => $factStat,
        ]);
    }

    public function update(UpdateFactStatRequest $request, FactStat $factStat)
    {
        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        $factStat->update($data);

        return redirect()
            ->route('admin.fact-stats.edit', $factStat)
            ->with('status', 'Fact updated successfully.');
    }

    public function destroy(FactStat $factStat)
    {
        $factStat->delete();

        return redirect()
            ->route('admin.fact-stats.index')
            ->with('status', 'Fact deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:fact_stats,id'],
        ]);

        FactStat::query()->whereIn('id', $validated['ids'])->delete();

        return redirect()
            ->route('admin.fact-stats.index')
            ->with('status', 'Selected facts deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = FactStat::query()->with('creator');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('fact', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('value', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $filename = 'fact-stats-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['fact', 'value', 'is_active', 'created_by', 'created_at']);

            $query->chunk(200, function ($facts) use ($handle) {
                foreach ($facts as $fact) {
                    fputcsv($handle, [
                        $fact->fact,
                        $fact->value,
                        $fact->is_active ? 1 : 0,
                        $fact->creator?->email,
                        optional($fact->created_at)->format('Y-m-d H:i:s'),
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportFactStatsRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.fact-stats.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.fact-stats.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, ['fact', 'value', 'is_active']);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $fact = trim((string) ($data['fact'] ?? ''));
            $value = trim((string) ($data['value'] ?? ''));

            if ($fact === '' || $value === '') {
                $skipped++;
                continue;
            }

            FactStat::updateOrCreate(
                ['fact' => $fact],
                [
                    'value' => $value,
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.fact-stats.index')
            ->with('status', "Imported {$imported} facts. Skipped {$skipped} rows.");
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'fact_asc' => $query->orderBy('fact'),
            'fact_desc' => $query->orderBy('fact', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }
}
