<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportFaqsRequest;
use App\Http\Requests\Admin\StoreFaqRequest;
use App\Http\Requests\Admin\UpdateFaqRequest;
use App\Models\Faq;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class FaqController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'order_asc');

        $query = Faq::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('question', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('answer', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return view('admin.faqs.index', [
            'faqs' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'status' => $status,
            'sort' => $sort,
        ]);
    }

    public function create()
    {
        return view('admin.faqs.create');
    }

    public function store(StoreFaqRequest $request)
    {
        $data = $request->validated();

        $faq = Faq::create([
            'question' => $data['question'],
            'slug' => $this->uniqueSlug($data['question']),
            'answer' => $data['answer'],
            'sort_order' => $data['sort_order'] ?? 0,
            'is_active' => $request->boolean('is_active'),
            'created_by' => $request->user()?->id,
        ]);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.faqs.create')
                ->with('status', 'FAQ created. You can add another.');
        }

        return redirect()
            ->route('admin.faqs.index')
            ->with('status', 'FAQ created successfully.');
    }

    public function show(Faq $faq)
    {
        return view('admin.faqs.show', [
            'faq' => $faq,
        ]);
    }

    public function edit(Faq $faq)
    {
        return view('admin.faqs.edit', [
            'faq' => $faq,
        ]);
    }

    public function update(UpdateFaqRequest $request, Faq $faq)
    {
        $data = $request->validated();

        $payload = [
            'question' => $data['question'],
            'answer' => $data['answer'],
            'sort_order' => $data['sort_order'] ?? 0,
            'is_active' => $request->boolean('is_active'),
        ];

        if ($data['question'] !== $faq->question) {
            $payload['slug'] = $this->uniqueSlug($data['question'], $faq->id);
        }

        $faq->update($payload);

        return redirect()
            ->route('admin.faqs.edit', $faq)
            ->with('status', 'FAQ updated successfully.');
    }

    public function destroy(Faq $faq)
    {
        $faq->delete();

        return redirect()
            ->route('admin.faqs.index')
            ->with('status', 'FAQ deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:faqs,id'],
        ]);

        Faq::query()->whereIn('id', $validated['ids'])->delete();

        return redirect()
            ->route('admin.faqs.index')
            ->with('status', 'Selected FAQs deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'order_asc');

        $query = Faq::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('question', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('answer', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $filename = 'faqs-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['question', 'answer', 'sort_order', 'is_active']);

            $query->chunk(200, function ($faqs) use ($handle) {
                foreach ($faqs as $faq) {
                    fputcsv($handle, [
                        $faq->question,
                        $faq->answer,
                        $faq->sort_order,
                        $faq->is_active ? 1 : 0,
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportFaqsRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.faqs.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.faqs.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, ['question', 'answer', 'sort_order', 'is_active']);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $question = trim((string) ($data['question'] ?? ''));
            $answer = trim((string) ($data['answer'] ?? ''));

            if ($question === '' || $answer === '') {
                $skipped++;
                continue;
            }

            $faq = Faq::query()->where('question', $question)->first();
            $payload = [
                'answer' => $answer,
                'sort_order' => $this->parseInt($data['sort_order'] ?? null) ?? 0,
                'is_active' => $this->parseBool($data['is_active'] ?? null, true),
            ];

            if ($faq) {
                $faq->update($payload);
            } else {
                Faq::create(array_merge($payload, [
                    'question' => $question,
                    'slug' => $this->uniqueSlug($question),
                ]));
            }

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.faqs.index')
            ->with('status', "Imported {$imported} FAQs. Skipped {$skipped} rows.");
    }

    private function parseInt($value): ?int
    {
        if ($value === null || trim((string) $value) === '') {
            return null;
        }

        return (int) $value;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'order_desc' => $query->orderBy('sort_order', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('sort_order'),
        };
    }

    private function uniqueSlug(string $question, ?int $ignoreId = null): string
    {
        $base = Str::slug($question);
        $slug = $base !== '' ? $base : Str::random(8);
        $counter = 1;

        while ($this->slugExists($slug, $ignoreId)) {
            $slug = $base . '-' . $counter;
            $counter++;
        }

        return $slug;
    }

    private function slugExists(string $slug, ?int $ignoreId): bool
    {
        $query = Faq::query()->where('slug', $slug);
        if ($ignoreId) {
            $query->where('id', '!=', $ignoreId);
        }

        return $query->exists();
    }
}
