<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Currency;
use App\Models\Order;
use App\Models\Report;
use App\Support\FormatSettings;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;

class FinancialReportController extends Controller
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', Report::class);

        $currencyConfig = FormatSettings::currencyConfig();
        $currencySymbol = $currencyConfig['symbol'];

        // Load exchange rates (same pattern as DashboardController)
        $currencyRates = Currency::query()
            ->where('is_active', true)
            ->pluck('exchange_rate', 'code')
            ->mapWithKeys(fn ($rate, $code) => [strtoupper($code) => (float) $rate])
            ->all();

        $defaultCode = strtoupper($currencyConfig['code'] ?? 'USD');
        $defaultRate = (float) ($currencyRates[$defaultCode] ?? 1.0);
        if ($defaultRate <= 0) {
            $defaultRate = 1.0;
        }

        // Determine date range from filters
        $year = (int) ($request->query('year') ?: now()->year);
        $startDate = $request->query('start_date');
        $endDate = $request->query('end_date');

        if ($startDate && $endDate) {
            try {
                $from = Carbon::parse($startDate)->startOfDay();
                $to = Carbon::parse($endDate)->endOfDay();
            } catch (\Throwable $e) {
                $from = Carbon::create($year, 1, 1)->startOfDay();
                $to = Carbon::create($year, 12, 31)->endOfDay();
            }
        } else {
            $from = Carbon::create($year, 1, 1)->startOfDay();
            $to = Carbon::create($year, 12, 31)->endOfDay();
        }

        // Currency converter closure
        $convertAmount = function ($order) use ($currencyRates, $defaultRate) {
            $orderRate = (float) ($currencyRates[strtoupper($order->currency ?? 'USD')] ?? 1.0);
            if ($orderRate <= 0) {
                $orderRate = 1.0;
            }

            return ($order->amount / $orderRate) * $defaultRate;
        };

        // Fetch completed orders in date range
        $completedOrders = Order::completed()
            ->whereBetween('created_at', [$from, $to])
            ->get(['amount', 'currency', 'created_at', 'plan_id', 'payment_gateway_id']);

        // Fetch refunded orders in date range
        $refundedOrders = Order::where('status', 'refunded')
            ->whereBetween('created_at', [$from, $to])
            ->get(['amount', 'currency', 'created_at']);

        // Summary calculations
        $totalRevenue = $completedOrders->sum($convertAmount);
        $totalRefunds = $refundedOrders->sum($convertAmount);
        $netRevenue = $totalRevenue - $totalRefunds;
        $totalOrders = $completedOrders->count();
        $avgOrderValue = $totalOrders > 0 ? $totalRevenue / $totalOrders : 0;

        // Monthly breakdown
        $months = [];
        for ($m = 1; $m <= 12; $m++) {
            $monthStart = Carbon::create($from->year <= $to->year ? ($m <= $to->month || $from->year < $to->year ? ($from->year === $to->year ? $year : ($m >= $from->month ? $from->year : $to->year)) : $year) : $year, $m, 1);

            // Simpler approach: iterate 1..12 for the selected year range
            $monthLabel = Carbon::create($year, $m, 1)->format('M Y');
            $monthKey = Carbon::create($year, $m, 1)->format('Y-m');

            $monthRevenue = $completedOrders
                ->filter(fn ($o) => $o->created_at->format('Y-m') === $monthKey)
                ->sum($convertAmount);

            $monthRefunds = $refundedOrders
                ->filter(fn ($o) => $o->created_at->format('Y-m') === $monthKey)
                ->sum($convertAmount);

            $monthOrderCount = $completedOrders
                ->filter(fn ($o) => $o->created_at->format('Y-m') === $monthKey)
                ->count();

            $months[] = [
                'label' => $monthLabel,
                'short' => Carbon::create($year, $m, 1)->format('M'),
                'revenue' => round($monthRevenue, 2),
                'refunds' => round($monthRefunds, 2),
                'net' => round($monthRevenue - $monthRefunds, 2),
                'orders' => $monthOrderCount,
                'avg' => $monthOrderCount > 0 ? round($monthRevenue / $monthOrderCount, 2) : 0,
            ];
        }

        // Revenue by plan (for donut chart)
        $planNames = \App\Models\Plan::pluck('name', 'id')->all();
        $revenueByPlan = $completedOrders
            ->groupBy('plan_id')
            ->map(fn ($group, $planId) => [
                'label' => $planNames[$planId] ?? 'Unknown',
                'value' => round($group->sum($convertAmount), 2),
            ])
            ->sortByDesc('value')
            ->values()
            ->all();

        // Revenue by gateway (for donut chart)
        $gatewayNames = \App\Models\PaymentGateway::pluck('name', 'id')->all();
        $revenueByGateway = $completedOrders
            ->groupBy('payment_gateway_id')
            ->map(fn ($group, $gwId) => [
                'label' => $gatewayNames[$gwId] ?? 'Direct',
                'value' => round($group->sum($convertAmount), 2),
            ])
            ->sortByDesc('value')
            ->values()
            ->all();

        return view('admin.financial-report.index', [
            'year' => $year,
            'startDate' => $startDate,
            'endDate' => $endDate,
            'currencySymbol' => $currencySymbol,
            'totalRevenue' => round($totalRevenue, 2),
            'totalRefunds' => round($totalRefunds, 2),
            'netRevenue' => round($netRevenue, 2),
            'totalOrders' => $totalOrders,
            'avgOrderValue' => round($avgOrderValue, 2),
            'months' => $months,
            'revenueByPlan' => $revenueByPlan,
            'revenueByGateway' => $revenueByGateway,
        ]);
    }
}
