<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Page;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class FrontendSettingsController extends Controller
{
    private array $pageConfig = [
        'about' => ['title' => 'About Us', 'slug' => 'about-us'],
        'contact' => ['title' => 'Contact Us', 'slug' => 'contact-us'],
        'privacy_policy' => ['title' => 'Privacy Policy', 'slug' => 'privacy-policy'],
        'terms' => ['title' => 'Terms and Conditions', 'slug' => 'terms-and-conditions'],
        'cookie_policy' => ['title' => 'Cookie Policy', 'slug' => 'cookie-policy'],
    ];

    public function index()
    {
        return view('admin.frontend-settings.index');
    }

    public function edit(string $type)
    {
        abort_unless(array_key_exists($type, $this->pageConfig), 404);

        $config = $this->pageConfig[$type];
        $page = Page::firstOrCreate(
            ['type' => $type],
            [
                'title' => $config['title'],
                'slug' => $config['slug'],
                'is_active' => true,
            ]
        );

        $content = array_merge($this->getDefaults($type), $page->content ?? []);

        return view('admin.frontend-settings.edit', [
            'page' => $page,
            'content' => $content,
            'type' => $type,
            'pageTitle' => $config['title'],
        ]);
    }

    public function update(Request $request, string $type)
    {
        abort_unless(array_key_exists($type, $this->pageConfig), 404);

        $config = $this->pageConfig[$type];
        $page = Page::firstOrCreate(
            ['type' => $type],
            [
                'title' => $config['title'],
                'slug' => $config['slug'],
                'is_active' => true,
            ]
        );

        $content = $request->input('content', []);

        $page->update([
            'content' => $content,
            'is_active' => (bool) $request->input('is_active', true),
            'updated_by' => $request->user()?->id,
        ]);

        return redirect()
            ->route('admin.frontend-settings.edit', $type)
            ->with('success', $config['title'] . ' settings updated successfully.');
    }

    private function getDefaults(string $type): array
    {
        $appName = config('app.name', 'SnapNest – Event Photo Sharing Platform with QR & Face Recognition');

        return match ($type) {
            'about' => [
                'badge_text' => 'Our Mission',
                'hero_title' => 'Redefining the',
                'hero_title_highlight' => 'Moment of Delivery.',
                'hero_description' => $appName . ' was born from a simple observation: photographers spend too much time on logistics and not enough time behind the lens. We\'re here to change that.',
                'stats' => [
                    ['label' => 'Photos Delivered', 'value' => '50M+'],
                    ['label' => 'Active Studios', 'value' => '2,000+'],
                    ['label' => 'Countries', 'value' => '45+'],
                    ['label' => 'Uptime', 'value' => '99.99%'],
                ],
                'story_title' => 'How it started.',
                'story_paragraph_1' => 'In 2022, our founders noticed a recurring problem at high-profile events: guests would wait days, even weeks, to receive their professional photos. The "hype" of the event had long passed by the time the gallery link arrived.',
                'story_paragraph_2' => 'We built ' . $appName . ' to bridge that gap. By leveraging advanced facial recognition and automated delivery workflows, we\'ve enabled photographers to share memories while they are still fresh—creating a "wow" moment for every guest.',
                'card_1_title' => 'Passion First',
                'card_1_description' => 'We are photographers at heart, building for photographers.',
                'card_2_title' => 'Pixel Perfect',
                'card_2_description' => 'Every feature is crafted with meticulous attention to detail.',
            ],
            'contact' => [
                'badge_text' => '24/7 Support Available',
                'hero_title' => "Let's start a",
                'hero_title_highlight' => 'Conversation.',
                'hero_description' => 'Have a specific query about our platform? Our team is ready to assist you in scaling your photography business.',
                'support_label' => 'Support',
                'support_text' => 'Live Chat - 24/7',
                'response_label' => 'Avg. Response',
                'response_value' => '15m',
                'sales_label' => 'Global Sales',
                'sales_value' => 'Active',
            ],
            'privacy_policy' => [
                'title' => 'Privacy',
                'title_highlight' => 'Matters.',
                'subtitle' => 'We believe privacy is a fundamental human right. Here is how we protect yours and your clients\'.',
                'last_updated' => 'Oct 2023',
                'sections' => [
                    ['title' => 'Information We Collect', 'content' => 'We collect information you provide directly to us when you create an account, create a gallery, or communicate with us. This includes your name, email address, studio name, and any images you upload for the purpose of event delivery.'],
                    ['title' => 'How We Use Face Search', 'content' => 'Our Face Search technology is designed purely for one-to-many matching within your private galleries. We do not build global face databases. Biometric data is stored securely and is only used to help guests find their photos at specific events.'],
                    ['title' => 'Data Security', 'content' => 'We use industry-standard encryption and security protocols to protect your data and your clients\' images. All photos are stored on redundant, secure cloud servers with strict access controls.'],
                    ['title' => 'Third-Party Sharing', 'content' => 'We do not sell your data. We only share information with third-party services (like WhatsApp or Cloud Providers) as necessary to provide our core services and ensure the best experience for your clients.'],
                ],
                'bottom_paragraph_1' => 'At ' . $appName . ', we understand the sensitive nature of photography. Whether it is a private wedding, a high-stakes corporate gathering, or a family function, we prioritize the confidentiality and security of every image uploaded to our platform.',
                'bottom_paragraph_2' => 'Your images are your property. We only process them to enable the features you\'ve selected, such as face matching for instant delivery. We do not use your images for marketing purposes without explicit consent, and we provide robust tools for users to request data deletion.',
            ],
            'terms' => [
                'title' => 'Terms of',
                'title_highlight' => 'Service',
                'last_updated' => 'January 1, 2026',
                'sections' => [
                    ['title' => 'Introduction', 'content' => 'Welcome to ' . $appName . '. These Terms of Service govern your use of the ' . $appName . ' platform, website, and associated services. By creating an account or using our platform, you agree to these terms in full.'],
                    ['title' => 'Services & Licensing', 'content' => $appName . ' provides a photography hosting and distribution platform powered by AI. We grant you a limited, non-exclusive license to use our platform for your professional photography business.'],
                    ['title' => 'User Responsibilities', 'content' => 'As a photographer on our platform, you are responsible for the content you upload. You must ensure you have the necessary rights to distribute the images and that your content does not violate any laws or third-party rights.'],
                    ['title' => 'AI Face Recognition', 'content' => 'Our AI features are designed to enhance guest experience. You agree to inform your clients and participants that AI indexing may be used for the purpose of photo delivery, in accordance with our Privacy Policy.'],
                    ['title' => 'Payment & Subscriptions', 'content' => 'Access to certain features requires a paid subscription. All fees are non-refundable unless otherwise specified. We reserve the right to change our pricing upon reasonable notice.'],
                    ['title' => 'Termination of Service', 'content' => 'We reserve the right to suspend or terminate accounts that violate these terms, engage in fraudulent activity, or disrupt the quality of service for other users.'],
                ],
            ],
            'cookie_policy' => [
                'title' => 'Cookie',
                'title_highlight' => 'Policy',
                'last_updated' => 'January 1, 2026',
                'intro_text' => 'At ' . $appName . ', we use cookies to provide a better experience and analyze our traffic. This policy explains what cookies are, how we use them, and your choices regarding their use.',
                'cookie_types' => [
                    ['title' => 'Essential Cookies', 'description' => 'These are necessary for the website to function. They include session management and security cookies.', 'priority' => 'Mandatory'],
                    ['title' => 'Performance Cookies', 'description' => 'We use these to understand how visitors interact with our platform, help us identify areas for improvement.', 'priority' => 'Optional'],
                    ['title' => 'Functionality Cookies', 'description' => 'These allow the website to remember choices you make, such as your language preference or layout settings.', 'priority' => 'Optional'],
                ],
                'manage_title' => 'How to Manage Cookies',
                'manage_text' => 'Most web browsers allow you to control cookies through their settings preferences. However, if you limit the ability of websites to set cookies, you may worsen your overall user experience, as it will no longer be personalized to you.',
            ],
            default => [],
        };
    }
}
