<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportLanguagesRequest;
use App\Http\Requests\Admin\StoreLanguageRequest;
use App\Http\Requests\Admin\UpdateLanguageRequest;
use App\Models\Language;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class LanguageController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $direction = $request->query('direction');
        $sort = $request->query('sort', 'name_asc');

        $query = Language::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('code', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('native_name', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($direction) {
            $query->where('direction', $direction);
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return view('admin.locations.languages.index', [
            'languages' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'status' => $status,
            'direction' => $direction,
            'sort' => $sort,
        ]);
    }

    public function create()
    {
        return view('admin.locations.languages.create');
    }

    public function store(StoreLanguageRequest $request)
    {
        $data = $request->validated();

        Language::create([
            'name' => $data['name'],
            'code' => $this->normalizeCode($data['code']),
            'native_name' => $this->normalizeText($data['native_name'] ?? null),
            'direction' => $this->normalizeDirection($data['direction'] ?? null),
            'is_active' => $request->boolean('is_active'),
        ]);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.languages.create')
                ->with('status', 'Language created. You can add another.');
        }

        return redirect()
            ->route('admin.languages.index')
            ->with('status', 'Language created successfully.');
    }

    public function show(Language $language)
    {
        return view('admin.locations.languages.show', [
            'language' => $language,
        ]);
    }

    public function edit(Language $language)
    {
        return view('admin.locations.languages.edit', [
            'language' => $language,
        ]);
    }

    public function update(UpdateLanguageRequest $request, Language $language)
    {
        $data = $request->validated();

        $language->update([
            'name' => $data['name'],
            'code' => $this->normalizeCode($data['code']),
            'native_name' => $this->normalizeText($data['native_name'] ?? null),
            'direction' => $this->normalizeDirection($data['direction'] ?? null),
            'is_active' => $request->boolean('is_active'),
        ]);

        return redirect()
            ->route('admin.languages.edit', $language)
            ->with('status', 'Language updated successfully.');
    }

    public function destroy(Language $language)
    {
        $language->delete();

        return redirect()
            ->route('admin.languages.index')
            ->with('status', 'Language deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:languages,id'],
        ]);

        Language::query()->whereIn('id', $validated['ids'])->delete();

        return redirect()
            ->route('admin.languages.index')
            ->with('status', 'Selected languages deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $direction = $request->query('direction');
        $sort = $request->query('sort', 'name_asc');

        $query = Language::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('code', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('native_name', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($direction) {
            $query->where('direction', $direction);
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $filename = 'languages-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['name', 'code', 'native_name', 'direction', 'is_active']);

            $query->chunk(200, function ($languages) use ($handle) {
                foreach ($languages as $language) {
                    fputcsv($handle, [
                        $language->name,
                        $language->code,
                        $language->native_name,
                        $language->direction,
                        $language->is_active ? 1 : 0,
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportLanguagesRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.languages.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.languages.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, ['name', 'code', 'native_name', 'direction', 'is_active']);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ''));
            $code = $this->normalizeCode($data['code'] ?? null);

            if ($name === '' || ! $code) {
                $skipped++;
                continue;
            }

            Language::updateOrCreate(
                ['code' => $code],
                [
                    'name' => $name,
                    'native_name' => $this->normalizeText($data['native_name'] ?? null),
                    'direction' => $this->normalizeDirection($data['direction'] ?? null),
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.languages.index')
            ->with('status', "Imported {$imported} languages. Skipped {$skipped} rows.");
    }

    private function normalizeCode(?string $value): ?string
    {
        if ($value === null) {
            return null;
        }

        $value = trim($value);

        return $value !== '' ? $value : null;
    }

    private function normalizeText(?string $value): ?string
    {
        if ($value === null) {
            return null;
        }

        $value = trim($value);

        return $value !== '' ? $value : null;
    }

    private function normalizeDirection(?string $value): string
    {
        $value = Str::lower(trim((string) $value));

        return $value === 'rtl' ? 'rtl' : 'ltr';
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_desc' => $query->orderBy('name', 'desc'),
            'code_asc' => $query->orderBy('code'),
            'code_desc' => $query->orderBy('code', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('name'),
        };
    }
}
