<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\NewsletterSubscriber;
use Illuminate\Http\Request;

class NewsletterSubscriberController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = NewsletterSubscriber::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('email', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('name', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return view('admin.newsletter-subscribers.index', [
            'subscribers' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'status' => $status,
            'sort' => $sort,
        ]);
    }

    public function show(NewsletterSubscriber $newsletterSubscriber)
    {
        return view('admin.newsletter-subscribers.show', [
            'subscriber' => $newsletterSubscriber,
        ]);
    }

    public function destroy(NewsletterSubscriber $newsletterSubscriber)
    {
        $newsletterSubscriber->delete();

        return redirect()
            ->route('admin.newsletter-subscribers.index')
            ->with('status', 'Subscriber deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:newsletter_subscribers,id'],
        ]);

        NewsletterSubscriber::query()->whereIn('id', $validated['ids'])->delete();

        return redirect()
            ->route('admin.newsletter-subscribers.index')
            ->with('status', 'Selected subscribers deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = NewsletterSubscriber::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('email', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('name', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $filename = 'newsletter-subscribers-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['email', 'name', 'is_active', 'subscribed_at', 'unsubscribed_at', 'created_at']);

            $query->chunk(200, function ($subscribers) use ($handle) {
                foreach ($subscribers as $subscriber) {
                    fputcsv($handle, [
                        $subscriber->email,
                        $subscriber->name,
                        $subscriber->is_active ? 1 : 0,
                        optional($subscriber->subscribed_at)->format('Y-m-d H:i:s'),
                        optional($subscriber->unsubscribed_at)->format('Y-m-d H:i:s'),
                        optional($subscriber->created_at)->format('Y-m-d H:i:s'),
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'email_asc' => $query->orderBy('email'),
            'email_desc' => $query->orderBy('email', 'desc'),
            'name_asc' => $query->orderBy('name'),
            'name_desc' => $query->orderBy('name', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }
}
