<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class OrderController extends Controller
{
    public function index(Request $request)
    {
        $user = auth()->user();
        $isAdmin = $user->hasRole(['Super Admin', 'Admin']);

        $query = Order::with(['user', 'plan'])->orderByDesc('created_at');

        if (!$isAdmin) {
            $query->where('user_id', $user->id);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('order_number', 'like', "%" . $this->escapeLike($search) . "%")
                  ->orWhereHas('user', function ($uq) use ($search) {
                      $uq->where('name', 'like', "%" . $this->escapeLike($search) . "%");
                  });
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $orders = $query->paginate(10)->withQueryString();

        return view('admin.orders.index', compact('orders', 'isAdmin'));
    }

    public function show(Order $order)
    {
        $user = auth()->user();
        $isAdmin = $user->hasRole(['Super Admin', 'Admin']);

        if (!$isAdmin && $order->user_id !== $user->id) {
            abort(403);
        }

        $order->load(['user', 'plan']);

        $setting = \App\Models\SystemSetting::first();
        $siteSettings = $setting?->payload['site'] ?? [];

        return view('admin.orders.invoice', compact('order', 'siteSettings'));
    }

    public function updateStatus(Request $request, Order $order)
    {
        $admin = auth()->user();
        if (! $admin->hasRole(['Super Admin', 'Admin'])) {
            abort(403);
        }

        $request->validate([
            'status' => ['required', Rule::in(['completed', 'pending', 'pending_verification', 'failed', 'refunded', 'cancelled'])],
            'note'   => ['nullable', 'string', 'max:500'],
        ]);

        $newStatus = $request->input('status');
        $note      = $request->input('note');
        $oldStatus = $order->status;

        $updateData = ['status' => $newStatus];

        // When manually completing an order, assign the plan to the user
        if ($newStatus === 'completed' && $oldStatus !== 'completed') {
            if ($order->plan_id) {
                $order->user()->update(['plan_id' => $order->plan_id]);
            }
        }

        // Append audit note to metadata
        if ($note || $oldStatus !== $newStatus) {
            $metadata = $order->metadata ?? [];
            $metadata['admin_updates'][] = [
                'admin'      => $admin->email,
                'from'       => $oldStatus,
                'to'         => $newStatus,
                'note'       => $note,
                'updated_at' => now()->toIso8601String(),
            ];
            $updateData['metadata'] = $metadata;
        }

        $order->update($updateData);

        return back()->with('success', "Order status updated from {$oldStatus} to {$newStatus}.");
    }

}
