<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportPagesRequest;
use App\Http\Requests\Admin\StorePageRequest;
use App\Http\Requests\Admin\UpdatePageRequest;
use App\Models\Page;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PageController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'title_asc');

        $query = Page::query()->customOnly();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('title', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('description', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return view('admin.pages.index', [
            'pages' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'status' => $status,
            'sort' => $sort,
            'types' => ['other' => 'Other'],
        ]);
    }

    public function create()
    {
        return view('admin.pages.create', [
            'types' => ['other' => 'Other'],
        ]);
    }

    public function store(StorePageRequest $request)
    {
        $data = $request->validated();

        Page::create([
            'title' => $data['title'],
            'slug' => $this->uniqueSlug($data['title']),
            'type' => $data['type'],
            'description' => $data['description'] ?? null,
            'is_active' => $request->boolean('is_active'),
            'created_by' => $request->user()?->id,
        ]);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.pages.create')
                ->with('status', 'Page created. You can add another.');
        }

        return redirect()
            ->route('admin.pages.index')
            ->with('status', 'Page created successfully.');
    }

    public function show(Page $page)
    {
        return view('admin.pages.show', [
            'page' => $page,
            'types' => ['other' => 'Other'],
        ]);
    }

    public function edit(Page $page)
    {
        return view('admin.pages.edit', [
            'page' => $page,
            'types' => ['other' => 'Other'],
        ]);
    }

    public function update(UpdatePageRequest $request, Page $page)
    {
        $data = $request->validated();

        $payload = [
            'title' => $data['title'],
            'type' => $data['type'],
            'description' => $data['description'] ?? null,
            'is_active' => $request->boolean('is_active'),
            'updated_by' => $request->user()?->id,
        ];

        if ($data['title'] !== $page->title) {
            $payload['slug'] = $this->uniqueSlug($data['title'], $page->id);
        }

        $page->update($payload);

        return redirect()
            ->route('admin.pages.edit', $page)
            ->with('status', 'Page updated successfully.');
    }

    public function destroy(Page $page)
    {
        $page->delete();

        return redirect()
            ->route('admin.pages.index')
            ->with('status', 'Page deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:pages,id'],
        ]);

        Page::query()->whereIn('id', $validated['ids'])->delete();

        return redirect()
            ->route('admin.pages.index')
            ->with('status', 'Selected pages deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'title_asc');

        $query = Page::query()->customOnly();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('title', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('description', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $filename = 'pages-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['title', 'slug', 'type', 'description', 'is_active']);

            $query->chunk(200, function ($pages) use ($handle) {
                foreach ($pages as $page) {
                    fputcsv($handle, [
                        $page->title,
                        $page->slug,
                        $page->type,
                        $page->description,
                        $page->is_active ? 1 : 0,
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportPagesRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.pages.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.pages.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, ['title', 'type', 'description', 'is_active']);
        $types = array_keys(config('system.page_types', []));

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $title = trim((string) ($data['title'] ?? ''));

            if ($title === '') {
                $skipped++;
                continue;
            }

            $type = $this->normalizeType($data['type'] ?? null, $types);

            $page = Page::query()->where('title', $title)->first();
            $payload = [
                'type' => $type,
                'description' => $data['description'] ?? null,
                'is_active' => $this->parseBool($data['is_active'] ?? null, true),
            ];

            if ($page) {
                $page->update($payload);
            } else {
                Page::create(array_merge($payload, [
                    'title' => $title,
                    'slug' => $this->uniqueSlug($title),
                ]));
            }

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.pages.index')
            ->with('status', "Imported {$imported} pages. Skipped {$skipped} rows.");
    }

    private function normalizeType(?string $value, array $types): string
    {
        $raw = trim((string) $value);
        $normalized = Str::slug($raw, '_');

        if ($normalized === 'terms_and_conditions') {
            $normalized = 'terms';
        }

        if (in_array($normalized, $types, true)) {
            return $normalized;
        }

        foreach (config('system.page_types', []) as $key => $label) {
            if (Str::lower($label) === Str::lower($raw)) {
                return $key;
            }
        }

        return 'other';
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'title_desc' => $query->orderBy('title', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('title'),
        };
    }

    private function uniqueSlug(string $title, ?int $ignoreId = null): string
    {
        $base = Str::slug($title);
        $slug = $base !== '' ? $base : Str::random(8);
        $counter = 1;

        while ($this->slugExists($slug, $ignoreId)) {
            $slug = $base . '-' . $counter;
            $counter++;
        }

        return $slug;
    }

    private function slugExists(string $slug, ?int $ignoreId): bool
    {
        $query = Page::query()->where('slug', $slug);
        if ($ignoreId) {
            $query->where('id', '!=', $ignoreId);
        }

        return $query->exists();
    }
}
