<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PaymentGateway;
use App\Services\Payment\PaymentGatewayManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;

class PaymentGatewayController extends Controller
{
    public function __construct(
        private readonly PaymentGatewayManager $gatewayManager,
    ) {}

    public function index()
    {
        $gateways = PaymentGateway::orderBy('sort_order')->get();

        // Attach credential field definitions and current values for each gateway
        $gateways->each(function (PaymentGateway $gateway) {
            $gateway->credentialFields = $this->gatewayManager->isSupported($gateway->slug)
                ? $this->gatewayManager->credentialFieldsFor($gateway->slug)
                : [];
            $gateway->decryptedCredentials = $gateway->getDecryptedCredentials();
        });

        return view('admin.payment-gateways.index', compact('gateways'));
    }

    public function update(Request $request, PaymentGateway $paymentGateway)
    {
        $request->validate([
            'mode' => ['required', 'in:sandbox,live'],
        ]);

        $credentialFields = $this->gatewayManager->isSupported($paymentGateway->slug)
            ? $this->gatewayManager->credentialFieldsFor($paymentGateway->slug)
            : [];

        $existingCredentials = $paymentGateway->getDecryptedCredentials();
        $newCredentials = [];

        foreach ($credentialFields as $field) {
            $key = $field['key'];
            $value = $request->input('credentials.' . $key);

            // For password fields, if the submitted value is empty and we already
            // have a stored value, keep the old value (don't overwrite with blank).
            if (($field['type'] ?? 'text') === 'password' && empty($value) && isset($existingCredentials[$key])) {
                $newCredentials[$key] = $existingCredentials[$key];
            } else {
                $newCredentials[$key] = $value ?? '';
            }
        }

        // Encrypt all credentials as JSON
        $paymentGateway->credentials = Crypt::encryptString(json_encode($newCredentials));

        // Update mode (sandbox/live)
        $paymentGateway->mode = $request->input('mode');

        $paymentGateway->save();

        return back()->with('status', $paymentGateway->name . ' settings saved successfully.');
    }

    public function toggleActive(PaymentGateway $paymentGateway)
    {
        $paymentGateway->update(['is_active' => ! $paymentGateway->is_active]);

        return back()->with('status', $paymentGateway->name . ($paymentGateway->is_active ? ' enabled' : ' disabled') . '.');
    }
}
