<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportPermissionsRequest;
use App\Http\Requests\Admin\StorePermissionRequest;
use App\Http\Requests\Admin\UpdatePermissionRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Spatie\Permission\Models\Permission;

class PermissionController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $guard = $request->query('guard');
        $sort = $request->query('sort', 'name_asc');

        $query = Permission::query();

        if ($search) {
            $query->where('name', 'like', "%" . $this->escapeLike($search) . "%");
        }

        if ($guard) {
            $query->where('guard_name', $guard);
        }

        $this->applySort($query, $sort);

        return view('admin.permissions.index', [
            'permissions' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'guard' => $guard,
            'sort' => $sort,
            'guards' => $this->availableGuards(),
        ]);
    }

    public function create()
    {
        return view('admin.permissions.create', [
            'guards' => $this->availableGuards(),
        ]);
    }

    public function store(StorePermissionRequest $request)
    {
        $data = $request->validated();

        Permission::create([
            'name' => $data['name'],
            'guard_name' => $data['guard_name'] ?? 'web',
        ]);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.permissions.create')
                ->with('status', 'Permission created. You can add another.');
        }

        return redirect()
            ->route('admin.permissions.index')
            ->with('status', 'Permission created successfully.');
    }

    public function show(Permission $permission)
    {
        $permission->loadCount('roles');

        return view('admin.permissions.show', [
            'permission' => $permission,
        ]);
    }

    public function edit(Permission $permission)
    {
        return view('admin.permissions.edit', [
            'permission' => $permission,
            'guards' => $this->availableGuards(),
        ]);
    }

    public function update(UpdatePermissionRequest $request, Permission $permission)
    {
        $data = $request->validated();

        $permission->update([
            'name' => $data['name'],
            'guard_name' => $data['guard_name'] ?? $permission->guard_name,
        ]);

        return redirect()
            ->route('admin.permissions.edit', $permission)
            ->with('status', 'Permission updated successfully.');
    }

    public function destroy(Permission $permission)
    {
        $permission->delete();

        return redirect()
            ->route('admin.permissions.index')
            ->with('status', 'Permission deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:permissions,id'],
        ]);

        Permission::query()->whereIn('id', $validated['ids'])->delete();

        return redirect()
            ->route('admin.permissions.index')
            ->with('status', 'Selected permissions deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $guard = $request->query('guard');
        $sort = $request->query('sort', 'name_asc');

        $query = Permission::query();

        if ($search) {
            $query->where('name', 'like', "%" . $this->escapeLike($search) . "%");
        }

        if ($guard) {
            $query->where('guard_name', $guard);
        }

        $this->applySort($query, $sort);

        $filename = 'permissions-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['name', 'guard_name']);

            $query->chunk(200, function ($permissions) use ($handle) {
                foreach ($permissions as $permission) {
                    fputcsv($handle, [
                        $permission->name,
                        $permission->guard_name,
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportPermissionsRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.permissions.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.permissions.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, ['name', 'guard_name']);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ''));
            $guardName = trim((string) ($data['guard_name'] ?? '')) ?: 'web';

            if ($name === '') {
                $skipped++;
                continue;
            }

            Permission::firstOrCreate([
                'name' => $name,
                'guard_name' => $guardName,
            ]);

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.permissions.index')
            ->with('status', "Imported {$imported} permissions. Skipped {$skipped} rows.");
    }

    private function availableGuards(): array
    {
        $guards = array_keys(config('auth.guards', []));

        return $guards ?: ['web'];
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_desc' => $query->orderBy('name', 'desc'),
            'guard_asc' => $query->orderBy('guard_name'),
            'guard_desc' => $query->orderBy('guard_name', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('name'),
        };
    }
}
