<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportPlansRequest;
use App\Http\Requests\Admin\StorePlanRequest;
use App\Http\Requests\Admin\UpdatePlanRequest;
use App\Models\Plan;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PlanController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = Plan::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return view('admin.plans.index', [
            'plans' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'status' => $status,
            'sort' => $sort,
        ]);
    }

    public function create()
    {
        return view('admin.plans.create');
    }

    public function store(StorePlanRequest $request)
    {
        $data = $request->validated();
        $features = $this->normalizeFeatures($data['features'] ?? null);

        $plan = Plan::create([
            'name' => $data['name'],
            'slug' => $this->uniqueSlug($data['name']),
            'price' => $data['price'],
            'offer_price' => $data['offer_price'] ?? null,
            'features' => $features,
            'images_limit' => $this->normalizeLimit($data['images_limit'] ?? null),
            'events_limit' => $this->normalizeLimit($data['events_limit'] ?? null),
            'storage_limit_gb' => isset($data['storage_limit_gb']) ? (float) $data['storage_limit_gb'] : null,
            'duration_type' => $data['duration_type'],
            'duration_months' => $data['duration_type'] === 'months' ? $data['duration_months'] : null,
            'has_watermark' => $request->boolean('has_watermark'),
            'has_design_editor' => $request->boolean('has_design_editor'),
            'videos_limit' => $this->normalizeLimit($data['videos_limit'] ?? null),
            'max_file_size_mb' => $this->normalizeLimit($data['max_file_size_mb'] ?? null),
            'has_guest_upload' => $request->boolean('has_guest_upload'),
            'has_google_drive_import' => $request->boolean('has_google_drive_import'),
            'has_ftp_import' => $request->boolean('has_ftp_import'),
            'has_custom_branding' => $request->boolean('has_custom_branding'),
            'has_cloud_storage' => $request->boolean('has_cloud_storage'),
            'is_active' => $request->boolean('is_active'),
            'created_by' => $request->user()?->id,
        ]);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.plans.create')
                ->with('status', 'Plan created. You can add another.');
        }

        return redirect()
            ->route('admin.plans.index')
            ->with('status', 'Plan created successfully.');
    }

    public function show(Plan $plan)
    {
        $plan->load('coupons');

        return view('admin.plans.show', [
            'plan' => $plan,
        ]);
    }

    public function edit(Plan $plan)
    {
        return view('admin.plans.edit', [
            'plan' => $plan,
        ]);
    }

    public function update(UpdatePlanRequest $request, Plan $plan)
    {
        $data = $request->validated();
        $features = $this->normalizeFeatures($data['features'] ?? null);

        $payload = [
            'name' => $data['name'],
            'price' => $data['price'],
            'offer_price' => $data['offer_price'] ?? null,
            'features' => $features,
            'images_limit' => $this->normalizeLimit($data['images_limit'] ?? null),
            'events_limit' => $this->normalizeLimit($data['events_limit'] ?? null),
            'storage_limit_gb' => isset($data['storage_limit_gb']) ? (float) $data['storage_limit_gb'] : null,
            'duration_type' => $data['duration_type'],
            'duration_months' => $data['duration_type'] === 'months' ? $data['duration_months'] : null,
            'has_watermark' => $request->boolean('has_watermark'),
            'has_design_editor' => $request->boolean('has_design_editor'),
            'videos_limit' => $this->normalizeLimit($data['videos_limit'] ?? null),
            'max_file_size_mb' => $this->normalizeLimit($data['max_file_size_mb'] ?? null),
            'has_guest_upload' => $request->boolean('has_guest_upload'),
            'has_google_drive_import' => $request->boolean('has_google_drive_import'),
            'has_ftp_import' => $request->boolean('has_ftp_import'),
            'has_custom_branding' => $request->boolean('has_custom_branding'),
            'has_cloud_storage' => $request->boolean('has_cloud_storage'),
            'is_active' => $request->boolean('is_active'),
        ];

        if ($data['name'] !== $plan->name) {
            $payload['slug'] = $this->uniqueSlug($data['name'], $plan->id);
        }

        $plan->update($payload);

        return redirect()
            ->route('admin.plans.edit', $plan)
            ->with('status', 'Plan updated successfully.');
    }

    public function destroy(Plan $plan)
    {
        $plan->coupons()->detach();
        $plan->delete();

        return redirect()
            ->route('admin.plans.index')
            ->with('status', 'Plan deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:plans,id'],
        ]);

        $plans = Plan::query()->whereIn('id', $validated['ids'])->get();
        foreach ($plans as $plan) {
            $plan->coupons()->detach();
            $plan->delete();
        }

        return redirect()
            ->route('admin.plans.index')
            ->with('status', 'Selected plans deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = Plan::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $filename = 'plans-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, [
                'name',
                'slug',
                'price',
                'offer_price',
                'features',
                'images_limit',
                'events_limit',
                'storage_limit_gb',
                'duration_type',
                'duration_months',
                'has_watermark',
                'has_design_editor',
                'videos_limit',
                'max_file_size_mb',
                'has_guest_upload',
                'has_google_drive_import',
                'has_ftp_import',
                'has_custom_branding',
                'has_cloud_storage',
                'is_active',
            ]);

            $query->chunk(200, function ($plans) use ($handle) {
                foreach ($plans as $plan) {
                    fputcsv($handle, [
                        $plan->name,
                        $plan->slug,
                        $plan->price,
                        $plan->offer_price,
                        $plan->features ? implode('|', $plan->features) : '',
                        $plan->images_limit,
                        $plan->events_limit,
                        $plan->storage_limit_gb,
                        $plan->duration_type,
                        $plan->duration_months,
                        $plan->has_watermark ? 1 : 0,
                        $plan->has_design_editor ? 1 : 0,
                        $plan->videos_limit,
                        $plan->max_file_size_mb,
                        $plan->has_guest_upload ? 1 : 0,
                        $plan->has_google_drive_import ? 1 : 0,
                        $plan->has_ftp_import ? 1 : 0,
                        $plan->has_custom_branding ? 1 : 0,
                        $plan->has_cloud_storage ? 1 : 0,
                        $plan->is_active ? 1 : 0,
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportPlansRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.plans.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.plans.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, [
            'name',
            'price',
            'offer_price',
            'features',
            'images_limit',
            'events_limit',
            'storage_limit_gb',
            'duration_type',
            'duration_months',
            'has_watermark',
            'has_design_editor',
            'videos_limit',
            'max_file_size_mb',
            'has_guest_upload',
            'has_google_drive_import',
            'has_ftp_import',
            'has_custom_branding',
            'has_cloud_storage',
            'is_active',
        ]);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ''));
            $price = $this->parseAmount($data['price'] ?? null);

            if ($name === '' || $price === null) {
                $skipped++;
                continue;
            }

            $durationType = $this->normalizeDurationType($data['duration_type'] ?? null);
            $durationMonths = $durationType === 'months'
                ? ($this->parseInt($data['duration_months'] ?? null) ?? 1)
                : null;

            $plan = Plan::query()->where('name', $name)->first();

            $payload = [
                'price' => $price,
                'offer_price' => $this->parseAmount($data['offer_price'] ?? null),
                'features' => $this->normalizeFeatures($data['features'] ?? null),
                'images_limit' => $this->normalizeLimit($this->parseInt($data['images_limit'] ?? null)),
                'events_limit' => $this->normalizeLimit($this->parseInt($data['events_limit'] ?? null)),
                'storage_limit_gb' => $this->parseAmount($data['storage_limit_gb'] ?? null),
                'duration_type' => $durationType,
                'duration_months' => $durationMonths,
                'has_watermark' => $this->parseBool($data['has_watermark'] ?? null, false),
                'has_design_editor' => $this->parseBool($data['has_design_editor'] ?? null, false),
                'videos_limit' => $this->normalizeLimit($this->parseInt($data['videos_limit'] ?? null)),
                'max_file_size_mb' => $this->normalizeLimit($this->parseInt($data['max_file_size_mb'] ?? null)),
                'has_guest_upload' => $this->parseBool($data['has_guest_upload'] ?? null, true),
                'has_google_drive_import' => $this->parseBool($data['has_google_drive_import'] ?? null, false),
                'has_ftp_import' => $this->parseBool($data['has_ftp_import'] ?? null, false),
                'has_custom_branding' => $this->parseBool($data['has_custom_branding'] ?? null, false),
                'has_cloud_storage' => $this->parseBool($data['has_cloud_storage'] ?? null, false),
                'is_active' => $this->parseBool($data['is_active'] ?? null, true),
            ];

            if ($plan) {
                $plan->update($payload);
            } else {
                Plan::create(array_merge($payload, [
                    'name' => $name,
                    'slug' => $this->uniqueSlug($name),
                ]));
            }

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.plans.index')
            ->with('status', "Imported {$imported} plans. Skipped {$skipped} rows.");
    }

    private function normalizeLimit($value): ?int
    {
        if ($value === null || $value === '') {
            return null;
        }

        $value = (int) $value;

        return $value >= 0 ? $value : null;
    }

    private function normalizeFeatures($features): ?array
    {
        if ($features === null) {
            return null;
        }

        $items = is_array($features)
            ? $features
            : preg_split('/[|,\n]/', (string) $features);

        $normalized = collect($items)
            ->map(fn ($item) => trim((string) $item))
            ->filter()
            ->unique()
            ->values()
            ->all();

        return $normalized ?: null;
    }

    private function normalizeDurationType(?string $value): string
    {
        if (! $value) {
            return 'months';
        }

        $value = Str::lower(trim($value));

        if (str_contains($value, 'life')) {
            return 'lifetime';
        }

        return 'months';
    }

    private function parseAmount($value): ?float
    {
        if ($value === null || trim((string) $value) === '') {
            return null;
        }

        return (float) $value;
    }

    private function parseInt($value): ?int
    {
        if ($value === null || trim((string) $value) === '') {
            return null;
        }

        return (int) $value;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_desc' => $query->orderBy('name', 'desc'),
            'price_asc' => $query->orderBy('price'),
            'price_desc' => $query->orderBy('price', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('name'),
        };
    }

    private function uniqueSlug(string $name, ?int $ignoreId = null): string
    {
        $base = Str::slug($name);
        $slug = $base !== '' ? $base : Str::random(8);
        $counter = 1;

        while ($this->slugExists($slug, $ignoreId)) {
            $slug = $base . '-' . $counter;
            $counter++;
        }

        return $slug;
    }

    private function slugExists(string $slug, ?int $ignoreId): bool
    {
        $query = Plan::query()->where('slug', $slug);
        if ($ignoreId) {
            $query->where('id', '!=', $ignoreId);
        }

        return $query->exists();
    }
}
