<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Coupon;
use App\Models\Event;
use App\Models\Plan;
use App\Models\SupportTicket;
use App\Models\User;
use App\Support\FormatSettings;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Spatie\Permission\Models\Role;

class ReportController extends Controller
{
    public function index()
    {
        $reports = [
            [
                'key' => 'users',
                'title' => 'User Report',
                'description' => 'Track account growth, verification, and assigned roles.',
                'total' => User::query()->count(),
            ],
            [
                'key' => 'events',
                'title' => 'Event Report',
                'description' => 'Review upcoming events, activity, and media volume.',
                'total' => Event::query()->count(),
            ],
            [
                'key' => 'plans',
                'title' => 'Plan Report',
                'description' => 'Monitor plan pricing, limits, and activation status.',
                'total' => Plan::query()->count(),
            ],
            [
                'key' => 'coupons',
                'title' => 'Coupon Report',
                'description' => 'Audit promo performance and validity windows.',
                'total' => Coupon::query()->count(),
            ],
            [
                'key' => 'support-tickets',
                'title' => 'Support Ticket Report',
                'description' => 'Track request volumes, priorities, and resolution status.',
                'total' => SupportTicket::query()->count(),
            ],
        ];

        return view('admin.reports.index', [
            'reports' => $reports,
            'analytics' => $this->getIndexAnalytics(),
        ]);
    }

    private function getIndexAnalytics(): array
    {
        $days = 30;
        $dates = [];
        for ($i = $days; $i >= 0; $i--) {
            $dates[] = now()->subDays($i)->format('Y-m-d');
        }

        $usersTrend = User::query()
            ->selectRaw('DATE(created_at) as date, count(*) as count')
            ->where('created_at', '>=', now()->subDays($days))
            ->groupBy('date')
            ->pluck('count', 'date');

        $eventsTrend = Event::query()
            ->selectRaw('DATE(created_at) as date, count(*) as count')
            ->where('created_at', '>=', now()->subDays($days))
            ->groupBy('date')
            ->pluck('count', 'date');

        return [
            'labels' => array_map(fn($date) => Carbon::parse($date)->format('M d'), $dates),
            'datasets' => [
                [
                    'name' => 'New Users',
                    'data' => array_map(fn($date) => $usersTrend[$date] ?? 0, $dates),
                ],
                [
                    'name' => 'New Events',
                    'data' => array_map(fn($date) => $eventsTrend[$date] ?? 0, $dates),
                ],
            ],
        ];
    }

    public function show(Request $request, string $report)
    {
        $definition = $this->definition($report);
        $query = ($definition['query'])();
        $filters = $this->applyFilters($report, $query, $request);

        // Save a clone for chart data before pagination
        $chartQuery = (clone $query)->reorder();

        $rows = $query->paginate(15)->withQueryString()->through($definition['map']);

        return view('admin.reports.show', [
            'reportKey' => $report,
            'report' => $definition,
            'filters' => $filters,
            'rows' => $rows,
            'charts' => $this->getReportCharts($report, $chartQuery),
        ]);
    }

    private function getReportCharts(string $report, Builder $query): array
    {
        $charts = [];

        switch ($report) {
            case 'users':
                $statusQuery = (clone $query)->setEagerLoads([]);
                $statusQuery->getQuery()->columns = [];
                $statusData = $statusQuery
                    ->selectRaw('CASE WHEN is_active = 1 THEN "Active" ELSE "Inactive" END as label, count(*) as value')
                    ->groupBy('is_active')
                    ->get();
                
                $charts[] = [
                    'id' => 'status-chart',
                    'title' => 'User Status Distribution',
                    'type' => 'pie',
                    'labels' => $statusData->pluck('label'),
                    'series' => $statusData->pluck('value'),
                ];

                $roleQuery = (clone $query)->setEagerLoads([])->reorder();
                $roleQuery->getQuery()->columns = [];
                $roleData = $roleQuery
                    ->leftJoin('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
                    ->leftJoin('roles', 'model_has_roles.role_id', '=', 'roles.id')
                    ->selectRaw('COALESCE(roles.name, "No Role") as label, count(*) as value')
                    ->groupBy('label')
                    ->get();

                $charts[] = [
                    'id' => 'role-chart',
                    'title' => 'Role Distribution',
                    'type' => 'donut',
                    'labels' => $roleData->pluck('label'),
                    'series' => $roleData->pluck('value'),
                ];
                break;

            case 'events':
                $typeQuery = (clone $query)->setEagerLoads([]);
                $typeQuery->getQuery()->columns = [];
                $typeData = $typeQuery
                    ->selectRaw('type as label, count(*) as value')
                    ->groupBy('type')
                    ->get();

                $charts[] = [
                    'id' => 'type-chart',
                    'title' => 'Event Types',
                    'type' => 'bar',
                    'labels' => $typeData->pluck('label'),
                    'series' => [['name' => 'Events', 'data' => $typeData->pluck('value')]],
                ];

                $statusQuery = (clone $query)->setEagerLoads([]);
                $statusQuery->getQuery()->columns = [];
                $statusData = $statusQuery
                    ->selectRaw('CASE WHEN is_active = 1 THEN "Active" ELSE "Inactive" END as label, count(*) as value')
                    ->groupBy('is_active')
                    ->get();

                $charts[] = [
                    'id' => 'status-chart',
                    'title' => 'Event Status',
                    'type' => 'pie',
                    'labels' => $statusData->pluck('label'),
                    'series' => $statusData->pluck('value'),
                ];
                break;

            case 'support-tickets':
                $statusQuery = (clone $query)->setEagerLoads([]);
                $statusQuery->getQuery()->columns = [];
                $statusData = $statusQuery
                    ->selectRaw('status as label, count(*) as value')
                    ->groupBy('status')
                    ->get();

                $charts[] = [
                    'id' => 'status-chart',
                    'title' => 'Ticket Status',
                    'type' => 'donut',
                    'labels' => $statusData->pluck('label')->map(fn($v) => $this->formatTitle($v)),
                    'series' => $statusData->pluck('value'),
                ];

                $priorityQuery = (clone $query)->setEagerLoads([]);
                $priorityQuery->getQuery()->columns = [];
                $priorityData = $priorityQuery
                    ->selectRaw('priority as label, count(*) as value')
                    ->groupBy('priority')
                    ->get();

                $charts[] = [
                    'id' => 'priority-chart',
                    'title' => 'Ticket Priority',
                    'type' => 'pie',
                    'labels' => $priorityData->pluck('label')->map(fn($v) => $this->formatTitle($v)),
                    'series' => $priorityData->pluck('value'),
                ];
                break;
            
            case 'plans':
                // 1. Duration types bar chart
                $durationQuery = (clone $query)->setEagerLoads([]);
                $durationQuery->getQuery()->columns = [];
                $durationData = $durationQuery
                    ->selectRaw('duration_type as label, count(*) as value')
                    ->groupBy('duration_type')
                    ->get();

                $charts[] = [
                    'id' => 'duration-chart',
                    'title' => 'Plan Duration Types',
                    'type' => 'bar',
                    'labels' => $durationData->pluck('label')->map(fn($v) => ucfirst($v)),
                    'series' => [['name' => 'Plans', 'data' => $durationData->pluck('value')]],
                ];

                // 2. Active vs Inactive pie chart
                $statusQuery = (clone $query)->setEagerLoads([]);
                $statusQuery->getQuery()->columns = [];
                $statusData = $statusQuery
                    ->selectRaw('CASE WHEN is_active = 1 THEN "Active" ELSE "Inactive" END as label, count(*) as value')
                    ->groupBy('is_active')
                    ->get();

                $charts[] = [
                    'id' => 'status-chart',
                    'title' => 'Plan Status',
                    'type' => 'pie',
                    'labels' => $statusData->pluck('label'),
                    'series' => $statusData->pluck('value'),
                ];

                // 3. Price comparison bar chart
                $priceQuery = (clone $query)->setEagerLoads([]);
                $priceQuery->getQuery()->columns = [];
                $priceData = $priceQuery
                    ->select('name', 'price', 'offer_price')
                    ->orderBy('price', 'asc')
                    ->get();

                if ($priceData->isNotEmpty()) {
                    $charts[] = [
                        'id' => 'price-chart',
                        'title' => 'Plan Pricing Comparison',
                        'type' => 'bar',
                        'labels' => $priceData->pluck('name'),
                        'series' => [
                            ['name' => 'Price', 'data' => $priceData->pluck('price')->map(fn($v) => (float) $v)],
                            ['name' => 'Offer Price', 'data' => $priceData->pluck('offer_price')->map(fn($v) => (float) ($v ?? 0))],
                        ],
                    ];
                }

                // 4. Feature availability horizontal bar
                $featureQuery = (clone $query)->setEagerLoads([]);
                $featureQuery->getQuery()->columns = [];
                $featureCounts = $featureQuery->selectRaw(
                    'SUM(has_watermark) as watermark, '
                    . 'SUM(has_design_editor) as design_editor, '
                    . 'SUM(has_guest_upload) as guest_upload, '
                    . 'SUM(has_google_drive_import) as google_drive, '
                    . 'SUM(has_ftp_import) as ftp_import, '
                    . 'SUM(has_custom_branding) as custom_branding, '
                    . 'SUM(has_cloud_storage) as cloud_storage'
                )->first();

                if ($featureCounts) {
                    $featureLabels = ['Watermark', 'Design Editor', 'Guest Upload', 'Google Drive', 'FTP Import', 'Custom Branding', 'Cloud Storage'];
                    $featureValues = [
                        (int) $featureCounts->watermark,
                        (int) $featureCounts->design_editor,
                        (int) $featureCounts->guest_upload,
                        (int) $featureCounts->google_drive,
                        (int) $featureCounts->ftp_import,
                        (int) $featureCounts->custom_branding,
                        (int) $featureCounts->cloud_storage,
                    ];

                    $charts[] = [
                        'id' => 'feature-chart',
                        'title' => 'Feature Availability Across Plans',
                        'type' => 'bar',
                        'labels' => $featureLabels,
                        'series' => [['name' => 'Plans with Feature', 'data' => $featureValues]],
                    ];
                }

                // 5. Subscribers per plan
                $subscriberData = User::query()
                    ->selectRaw('plans.name as label, count(users.id) as value')
                    ->join('plans', 'users.plan_id', '=', 'plans.id')
                    ->groupBy('plans.id', 'plans.name')
                    ->orderByDesc('value')
                    ->get();

                if ($subscriberData->isNotEmpty()) {
                    $charts[] = [
                        'id' => 'subscriber-chart',
                        'title' => 'Subscribers per Plan',
                        'type' => 'bar',
                        'labels' => $subscriberData->pluck('label'),
                        'series' => [['name' => 'Subscribers', 'data' => $subscriberData->pluck('value')]],
                    ];
                }

                // 6. Storage limits donut
                $storageQuery = (clone $query)->setEagerLoads([]);
                $storageQuery->getQuery()->columns = [];
                $storageData = $storageQuery
                    ->selectRaw('CASE '
                        . 'WHEN storage_limit_gb IS NULL THEN "Unlimited" '
                        . 'WHEN storage_limit_gb <= 5 THEN "Up to 5 GB" '
                        . 'WHEN storage_limit_gb <= 20 THEN "5-20 GB" '
                        . 'WHEN storage_limit_gb <= 50 THEN "20-50 GB" '
                        . 'ELSE "50+ GB" END as label, count(*) as value')
                    ->groupByRaw('CASE '
                        . 'WHEN storage_limit_gb IS NULL THEN "Unlimited" '
                        . 'WHEN storage_limit_gb <= 5 THEN "Up to 5 GB" '
                        . 'WHEN storage_limit_gb <= 20 THEN "5-20 GB" '
                        . 'WHEN storage_limit_gb <= 50 THEN "20-50 GB" '
                        . 'ELSE "50+ GB" END')
                    ->get();

                if ($storageData->isNotEmpty()) {
                    $charts[] = [
                        'id' => 'storage-chart',
                        'title' => 'Storage Limit Distribution',
                        'type' => 'donut',
                        'labels' => $storageData->pluck('label'),
                        'series' => $storageData->pluck('value'),
                    ];
                }
                break;

            case 'coupons':
                $typeQuery = (clone $query)->setEagerLoads([]);
                $typeQuery->getQuery()->columns = [];
                $typeData = $typeQuery
                    ->selectRaw('discount_type as label, count(*) as value')
                    ->groupBy('discount_type')
                    ->get();

                $charts[] = [
                    'id' => 'type-chart',
                    'title' => 'Discount Types',
                    'type' => 'donut',
                    'labels' => $typeData->pluck('label')->map(fn($v) => ucfirst($v)),
                    'series' => $typeData->pluck('value'),
                ];
                break;
        }

        return $charts;
    }

    public function export(Request $request, string $report, string $format)
    {
        $definition = $this->definition($report);
        $format = strtolower($format);

        if (! in_array($format, ['csv', 'xlsx', 'pdf'], true)) {
            abort(404);
        }

        $query = ($definition['query'])();
        $this->applyFilters($report, $query, $request);

        $filename = $report . '-' . now()->format('Ymd-His');

        if ($format === 'csv') {
            return $this->exportCsv($definition, $query, $filename . '.csv');
        }

        $rows = $query->get()->map($definition['map']);

        if ($format === 'xlsx') {
            return $this->exportXlsx($definition, $rows, $filename . '.xlsx');
        }

        return $this->exportPdf($definition, $rows, $filename . '.pdf');
    }

    private function definition(string $report): array
    {
        $definitions = $this->definitions();

        if (! array_key_exists($report, $definitions)) {
            abort(404);
        }

        return $definitions[$report];
    }

    private function definitions(): array
    {
        return [
            'users' => [
                'title' => 'User Report',
                'subtitle' => 'Monitor account status, verification, and role coverage.',
                'description' => 'Analyze user growth with status and role distribution.',
                'empty' => 'No users match the selected filters.',
                'columns' => [
                    ['key' => 'name', 'label' => 'Name'],
                    ['key' => 'email', 'label' => 'Email'],
                    ['key' => 'phone', 'label' => 'Phone'],
                    ['key' => 'roles', 'label' => 'Roles'],
                    ['key' => 'status', 'label' => 'Status'],
                    ['key' => 'verified', 'label' => 'Verified'],
                    ['key' => 'created_at', 'label' => 'Joined'],
                ],
                'query' => fn () => User::query()->with('roles')->orderBy('users.created_at', 'desc'),
                'map' => fn (User $user) => [
                    'name' => $user->name,
                    'email' => $user->email,
                    'phone' => $user->phone ?? '-',
                    'roles' => $user->roles->pluck('name')->implode(', ') ?: 'None',
                    'status' => $user->is_active ? 'Active' : 'Inactive',
                    'verified' => $user->email_verified_at ? 'Yes' : 'No',
                    'created_at' => FormatSettings::date($user->created_at),
                ],
            ],
            'events' => [
                'title' => 'Event Report',
                'subtitle' => 'Track event timelines, privacy status, and uploads.',
                'description' => 'Review event types, dates, and media volume.',
                'empty' => 'No events match the selected filters.',
                'columns' => [
                    ['key' => 'name', 'label' => 'Event'],
                    ['key' => 'type', 'label' => 'Type'],
                    ['key' => 'event_date', 'label' => 'Event Date'],
                    ['key' => 'expiry_date', 'label' => 'Expiry Date'],
                    ['key' => 'media_count', 'label' => 'Media'],
                    ['key' => 'status', 'label' => 'Status'],
                    ['key' => 'created_by', 'label' => 'Created By'],
                ],
                'query' => fn () => Event::query()
                    ->with('createdBy')
                    ->withCount('media')
                    ->orderBy('event_date', 'desc'),
                'map' => fn (Event $event) => [
                    'name' => $event->name,
                    'type' => $event->type,
                    'event_date' => FormatSettings::date($event->event_date),
                    'expiry_date' => FormatSettings::date($event->expiry_date),
                    'media_count' => (string) $event->media_count,
                    'status' => $event->is_active ? 'Active' : 'Inactive',
                    'created_by' => $event->createdBy?->name ?? 'System',
                ],
            ],
            'plans' => [
                'title' => 'Plan Report',
                'subtitle' => 'Review pricing, durations, and included limits.',
                'description' => 'Track active plan offerings and pricing structure.',
                'empty' => 'No plans match the selected filters.',
                'columns' => [
                    ['key' => 'name', 'label' => 'Plan'],
                    ['key' => 'price', 'label' => 'Price'],
                    ['key' => 'offer_price', 'label' => 'Offer'],
                    ['key' => 'duration', 'label' => 'Duration'],
                    ['key' => 'images_limit', 'label' => 'Image Limit'],
                    ['key' => 'events_limit', 'label' => 'Event Limit'],
                    ['key' => 'status', 'label' => 'Status'],
                ],
                'query' => fn () => Plan::query()->orderBy('created_at', 'desc'),
                'map' => fn (Plan $plan) => [
                    'name' => $plan->name,
                    'price' => $plan->price == 0 ? 'Free' : FormatSettings::currency($plan->price),
                    'offer_price' => $plan->offer_price !== null ? FormatSettings::currency($plan->offer_price) : '-',
                    'duration' => $plan->duration_type === 'lifetime'
                        ? 'Lifetime'
                        : ($plan->duration_months ? $plan->duration_months . ' months' : 'Monthly'),
                    'images_limit' => $plan->images_limit !== null ? (string) $plan->images_limit : 'Unlimited',
                    'events_limit' => $plan->events_limit !== null ? (string) $plan->events_limit : 'Unlimited',
                    'status' => $plan->is_active ? 'Active' : 'Inactive',
                ],
            ],
            'coupons' => [
                'title' => 'Coupon Report',
                'subtitle' => 'Audit promo coverage, timing, and eligibility.',
                'description' => 'Track discount codes and validation windows.',
                'empty' => 'No coupons match the selected filters.',
                'columns' => [
                    ['key' => 'name', 'label' => 'Coupon'],
                    ['key' => 'code', 'label' => 'Code'],
                    ['key' => 'discount_type', 'label' => 'Type'],
                    ['key' => 'discount_value', 'label' => 'Discount'],
                    ['key' => 'min_amount', 'label' => 'Min Amount'],
                    ['key' => 'starts_at', 'label' => 'Starts'],
                    ['key' => 'ends_at', 'label' => 'Ends'],
                    ['key' => 'new_users_only', 'label' => 'New Users'],
                    ['key' => 'status', 'label' => 'Status'],
                ],
                'query' => fn () => Coupon::query()->orderBy('created_at', 'desc'),
                'map' => fn (Coupon $coupon) => [
                    'name' => $coupon->name,
                    'code' => $coupon->code,
                    'discount_type' => $coupon->discount_type === 'percent' ? 'Percent' : 'Fixed',
                    'discount_value' => $coupon->discount_type === 'percent'
                        ? rtrim(rtrim(number_format($coupon->discount_value, 2), '0'), '.') . '%'
                        : FormatSettings::currency($coupon->discount_value),
                    'min_amount' => FormatSettings::currency($coupon->min_amount),
                    'starts_at' => FormatSettings::date($coupon->starts_at),
                    'ends_at' => FormatSettings::date($coupon->ends_at),
                    'new_users_only' => $coupon->new_users_only ? 'Yes' : 'No',
                    'status' => $coupon->is_active ? 'Active' : 'Inactive',
                ],
            ],
            'support-tickets' => [
                'title' => 'Support Ticket Report',
                'subtitle' => 'Monitor incoming requests and response timelines.',
                'description' => 'Review status and assignment coverage.',
                'empty' => 'No tickets match the selected filters.',
                'columns' => [
                    ['key' => 'ticket_number', 'label' => 'Ticket'],
                    ['key' => 'subject', 'label' => 'Subject'],
                    ['key' => 'priority', 'label' => 'Priority'],
                    ['key' => 'status', 'label' => 'Status'],
                    ['key' => 'requester', 'label' => 'Requester'],
                    ['key' => 'requester_email', 'label' => 'Email'],
                    ['key' => 'assigned_to', 'label' => 'Assigned To'],
                    ['key' => 'created_at', 'label' => 'Created'],
                    ['key' => 'resolved_at', 'label' => 'Resolved'],
                ],
                'query' => fn () => SupportTicket::query()
                    ->with('currentAssignment.assignedTo')
                    ->orderBy('created_at', 'desc'),
                'map' => fn (SupportTicket $ticket) => [
                    'ticket_number' => $ticket->ticket_number,
                    'subject' => $ticket->subject,
                    'priority' => $this->formatTitle($ticket->priority),
                    'status' => $this->formatTitle($ticket->status),
                    'requester' => $ticket->requester_name,
                    'requester_email' => $ticket->requester_email,
                    'assigned_to' => $ticket->currentAssignment?->assignedTo?->name ?? 'Unassigned',
                    'created_at' => FormatSettings::date($ticket->created_at),
                    'resolved_at' => FormatSettings::dateTime($ticket->resolved_at),
                ],
            ],
        ];
    }

    private function applyFilters(string $report, Builder $query, Request $request): array
    {
        $search = trim((string) $request->query('search', ''));
        $status = $request->query('status');
        $startDate = $this->normalizeDate($request->query('start_date'));
        $endDate = $this->normalizeDate($request->query('end_date'));

        $fields = [];

        switch ($report) {
            case 'users':
                $role = $request->query('role');

                if ($search !== '') {
                    $query->where(function ($builder) use ($search) {
                        $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('email', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('phone', 'like', "%" . $this->escapeLike($search) . "%");
                    });
                }

                if ($status === 'active') {
                    $query->where('is_active', true);
                } elseif ($status === 'inactive') {
                    $query->where('is_active', false);
                }

                if ($role) {
                    $query->whereHas('roles', function ($builder) use ($role) {
                        $builder->where('id', $role);
                    });
                }

                $this->applyDateRange($query, 'users.created_at', $startDate, $endDate);

                $roles = Role::query()->orderBy('name')->get()->pluck('name', 'id')->all();
                $roleOptions = array_merge(['' => 'All roles'], $roles);

                $fields = [
                    [
                        'name' => 'search',
                        'type' => 'text',
                        'label' => 'Search',
                        'placeholder' => 'Search by name, email, or phone',
                        'value' => $search,
                    ],
                    [
                        'name' => 'role',
                        'type' => 'select',
                        'label' => 'Role',
                        'options' => $roleOptions,
                        'value' => $role,
                    ],
                    [
                        'name' => 'status',
                        'type' => 'select',
                        'label' => 'Status',
                        'options' => [
                            '' => 'All statuses',
                            'active' => 'Active',
                            'inactive' => 'Inactive',
                        ],
                        'value' => $status,
                    ],
                    [
                        'name' => 'start_date',
                        'type' => 'date',
                        'label' => 'From',
                        'value' => $startDate,
                    ],
                    [
                        'name' => 'end_date',
                        'type' => 'date',
                        'label' => 'To',
                        'value' => $endDate,
                    ],
                ];
                break;
            case 'events':
                $type = $request->query('type');

                if ($search !== '') {
                    $query->where(function ($builder) use ($search) {
                        $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%");
                    });
                }

                if ($type) {
                    $query->where('type', $type);
                }

                if ($status === 'active') {
                    $query->where('is_active', true);
                } elseif ($status === 'inactive') {
                    $query->where('is_active', false);
                }

                $this->applyDateRange($query, 'event_date', $startDate, $endDate);

                $typeOptions = ['' => 'All types'];
                foreach (config('events.types', []) as $eventType) {
                    $typeOptions[$eventType] = $eventType;
                }

                $fields = [
                    [
                        'name' => 'search',
                        'type' => 'text',
                        'label' => 'Search',
                        'placeholder' => 'Search by event name or slug',
                        'value' => $search,
                    ],
                    [
                        'name' => 'type',
                        'type' => 'select',
                        'label' => 'Type',
                        'options' => $typeOptions,
                        'value' => $type,
                    ],
                    [
                        'name' => 'status',
                        'type' => 'select',
                        'label' => 'Status',
                        'options' => [
                            '' => 'All statuses',
                            'active' => 'Active',
                            'inactive' => 'Inactive',
                        ],
                        'value' => $status,
                    ],
                    [
                        'name' => 'start_date',
                        'type' => 'date',
                        'label' => 'From',
                        'value' => $startDate,
                    ],
                    [
                        'name' => 'end_date',
                        'type' => 'date',
                        'label' => 'To',
                        'value' => $endDate,
                    ],
                ];
                break;
            case 'plans':
                if ($search !== '') {
                    $query->where(function ($builder) use ($search) {
                        $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%");
                    });
                }

                if ($status === 'active') {
                    $query->where('is_active', true);
                } elseif ($status === 'inactive') {
                    $query->where('is_active', false);
                }

                $this->applyDateRange($query, 'created_at', $startDate, $endDate);

                $fields = [
                    [
                        'name' => 'search',
                        'type' => 'text',
                        'label' => 'Search',
                        'placeholder' => 'Search by plan name or slug',
                        'value' => $search,
                    ],
                    [
                        'name' => 'status',
                        'type' => 'select',
                        'label' => 'Status',
                        'options' => [
                            '' => 'All statuses',
                            'active' => 'Active',
                            'inactive' => 'Inactive',
                        ],
                        'value' => $status,
                    ],
                    [
                        'name' => 'start_date',
                        'type' => 'date',
                        'label' => 'From',
                        'value' => $startDate,
                    ],
                    [
                        'name' => 'end_date',
                        'type' => 'date',
                        'label' => 'To',
                        'value' => $endDate,
                    ],
                ];
                break;
            case 'coupons':
                $discountType = $request->query('discount_type');

                if ($search !== '') {
                    $query->where(function ($builder) use ($search) {
                        $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('code', 'like', "%" . $this->escapeLike($search) . "%");
                    });
                }

                if ($discountType) {
                    $query->where('discount_type', $discountType);
                }

                if ($status === 'active') {
                    $query->where('is_active', true);
                } elseif ($status === 'inactive') {
                    $query->where('is_active', false);
                }

                if ($startDate) {
                    $query->whereDate('starts_at', '>=', $startDate);
                }

                if ($endDate) {
                    $query->whereDate('ends_at', '<=', $endDate);
                }

                $fields = [
                    [
                        'name' => 'search',
                        'type' => 'text',
                        'label' => 'Search',
                        'placeholder' => 'Search by coupon name or code',
                        'value' => $search,
                    ],
                    [
                        'name' => 'discount_type',
                        'type' => 'select',
                        'label' => 'Type',
                        'options' => [
                            '' => 'All types',
                            'percent' => 'Percent',
                            'fixed' => 'Fixed',
                        ],
                        'value' => $discountType,
                    ],
                    [
                        'name' => 'status',
                        'type' => 'select',
                        'label' => 'Status',
                        'options' => [
                            '' => 'All statuses',
                            'active' => 'Active',
                            'inactive' => 'Inactive',
                        ],
                        'value' => $status,
                    ],
                    [
                        'name' => 'start_date',
                        'type' => 'date',
                        'label' => 'Valid From',
                        'value' => $startDate,
                    ],
                    [
                        'name' => 'end_date',
                        'type' => 'date',
                        'label' => 'Valid To',
                        'value' => $endDate,
                    ],
                ];
                break;
            case 'support-tickets':
                $priority = $request->query('priority');
                $statuses = config('support.statuses', []);
                $priorities = config('support.priorities', []);

                if ($search !== '') {
                    $query->where(function ($builder) use ($search) {
                        $builder->where('ticket_number', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('subject', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('requester_name', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('requester_email', 'like', "%" . $this->escapeLike($search) . "%");
                    });
                }

                if ($status) {
                    $query->where('status', $status);
                }

                if ($priority) {
                    $query->where('priority', $priority);
                }

                $this->applyDateRange($query, 'created_at', $startDate, $endDate);

                $fields = [
                    [
                        'name' => 'search',
                        'type' => 'text',
                        'label' => 'Search',
                        'placeholder' => 'Search by ticket, subject, or requester',
                        'value' => $search,
                    ],
                    [
                        'name' => 'status',
                        'type' => 'select',
                        'label' => 'Status',
                        'options' => array_merge(['' => 'All statuses'], $statuses),
                        'value' => $status,
                    ],
                    [
                        'name' => 'priority',
                        'type' => 'select',
                        'label' => 'Priority',
                        'options' => array_merge(['' => 'All priorities'], $priorities),
                        'value' => $priority,
                    ],
                    [
                        'name' => 'start_date',
                        'type' => 'date',
                        'label' => 'From',
                        'value' => $startDate,
                    ],
                    [
                        'name' => 'end_date',
                        'type' => 'date',
                        'label' => 'To',
                        'value' => $endDate,
                    ],
                ];
                break;
            default:
                $fields = [];
        }

        return [
            'fields' => $fields,
            'reset' => route('admin.reports.show', $report),
        ];
    }

    private function applyDateRange(Builder $query, string $column, ?string $startDate, ?string $endDate): void
    {
        if ($startDate) {
            $query->whereDate($column, '>=', $startDate);
        }

        if ($endDate) {
            $query->whereDate($column, '<=', $endDate);
        }
    }

    private function normalizeDate(?string $value): ?string
    {
        if (! $value) {
            return null;
        }

        try {
            return Carbon::parse($value)->format('Y-m-d');
        } catch (\Throwable $exception) {
            return null;
        }
    }

    private function exportCsv(array $definition, Builder $query, string $filename)
    {
        $columns = $definition['columns'];
        $map = $definition['map'];

        return response()->streamDownload(function () use ($query, $columns, $map) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, array_map(fn ($column) => $column['label'], $columns));

            $query->chunkById(200, function ($items) use ($handle, $columns, $map) {
                foreach ($items as $item) {
                    $row = $map($item);
                    fputcsv($handle, $this->formatRow($columns, $row));
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    private function exportXlsx(array $definition, Collection $rows, string $filename)
    {
        $columns = $definition['columns'];
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $columnIndex = 1;
        foreach ($columns as $column) {
            $sheet->setCellValueByColumnAndRow($columnIndex, 1, $column['label']);
            $columnIndex++;
        }

        $rowIndex = 2;
        foreach ($rows as $row) {
            $columnIndex = 1;
            foreach ($columns as $column) {
                $sheet->setCellValueByColumnAndRow(
                    $columnIndex,
                    $rowIndex,
                    $row[$column['key']] ?? ''
                );
                $columnIndex++;
            }
            $rowIndex++;
        }

        $writer = new Xlsx($spreadsheet);

        return response()->streamDownload(function () use ($writer) {
            $writer->save('php://output');
        }, $filename, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    private function exportPdf(array $definition, Collection $rows, string $filename)
    {
        $pdf = Pdf::loadView('admin.reports.pdf', [
            'report' => $definition,
            'columns' => $definition['columns'],
            'rows' => $rows,
            'generatedAt' => now(),
        ])->setPaper('a4', 'landscape');

        return $pdf->download($filename);
    }

    private function formatRow(array $columns, array $row): array
    {
        return array_map(function ($column) use ($row) {
            return $row[$column['key']] ?? '';
        }, $columns);
    }

    private function formatTitle(?string $value): string
    {
        if (! $value) {
            return '-';
        }

        return str_replace('_', ' ', ucwords($value, '_'));
    }
}
