<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportRolesRequest;
use App\Http\Requests\Admin\StoreRoleRequest;
use App\Http\Requests\Admin\UpdateRoleRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

class RoleController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $guard = $request->query('guard');

        $query = Role::query()
            ->withCount(['permissions', 'users'])
            ->orderBy('name');

        if ($search) {
            $query->where('name', 'like', "%" . $this->escapeLike($search) . "%");
        }

        if ($guard) {
            $query->where('guard_name', $guard);
        }

        return view('admin.roles.index', [
            'roles' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'guard' => $guard,
            'guards' => $this->availableGuards(),
        ]);
    }

    public function create()
    {
        $permissions = Permission::query()->orderBy('name')->get();

        return view('admin.roles.create', [
            'guards' => $this->availableGuards(),
            'permissionGroups' => $this->groupPermissions($permissions),
        ]);
    }

    public function store(StoreRoleRequest $request)
    {
        $data = $request->validated();
        $guard = $data['guard_name'] ?? 'web';

        $role = Role::create([
            'name' => $data['name'],
            'guard_name' => $guard,
        ]);

        $permissionIds = $this->filterPermissionsByGuard($data['permissions'] ?? [], $guard);
        $role->syncPermissions($permissionIds);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.roles.create')
                ->with('status', 'Role created. You can add another.');
        }

        return redirect()
            ->route('admin.roles.index')
            ->with('status', 'Role created successfully.');
    }

    public function show(Role $role)
    {
        $role->load(['permissions' => function ($query) {
            $query->orderBy('name');
        }, 'users']);

        return view('admin.roles.show', [
            'role' => $role,
        ]);
    }

    public function edit(Role $role)
    {
        $permissions = Permission::query()->orderBy('name')->get();

        return view('admin.roles.edit', [
            'role' => $role,
            'guards' => $this->availableGuards(),
            'permissionGroups' => $this->groupPermissions($permissions),
        ]);
    }

    public function update(UpdateRoleRequest $request, Role $role)
    {
        $data = $request->validated();
        $guard = $data['guard_name'] ?? $role->guard_name;

        $role->update([
            'name' => $data['name'],
            'guard_name' => $guard,
        ]);

        $permissionIds = $this->filterPermissionsByGuard($data['permissions'] ?? [], $guard);
        $role->syncPermissions($permissionIds);

        return redirect()
            ->route('admin.roles.edit', $role)
            ->with('status', 'Role updated successfully.');
    }

    public function destroy(Role $role)
    {
        $role->syncPermissions([]);
        $role->delete();

        return redirect()
            ->route('admin.roles.index')
            ->with('status', 'Role deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:roles,id'],
        ]);

        $roles = Role::query()->whereIn('id', $validated['ids'])->get();
        foreach ($roles as $role) {
            $role->syncPermissions([]);
            $role->delete();
        }

        return redirect()
            ->route('admin.roles.index')
            ->with('status', 'Selected roles deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $guard = $request->query('guard');

        $query = Role::query()->with('permissions')->orderBy('name');

        if ($search) {
            $query->where('name', 'like', "%" . $this->escapeLike($search) . "%");
        }

        if ($guard) {
            $query->where('guard_name', $guard);
        }

        $filename = 'roles-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['name', 'guard_name', 'permissions']);

            $query->chunk(200, function ($roles) use ($handle) {
                foreach ($roles as $role) {
                    fputcsv($handle, [
                        $role->name,
                        $role->guard_name,
                        $role->permissions->pluck('name')->implode('|'),
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportRolesRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.roles.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.roles.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, ['name', 'guard_name', 'permissions']);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ''));
            $guardName = trim((string) ($data['guard_name'] ?? '')) ?: 'web';

            if ($name === '') {
                $skipped++;
                continue;
            }

            $role = Role::firstOrCreate([
                'name' => $name,
                'guard_name' => $guardName,
            ]);

            $permissionNames = $this->parsePermissionNames($data['permissions'] ?? null);
            $permissionIds = [];

            foreach ($permissionNames as $permissionName) {
                $permission = Permission::firstOrCreate([
                    'name' => $permissionName,
                    'guard_name' => $guardName,
                ]);
                $permissionIds[] = $permission->id;
            }

            if ($permissionIds) {
                $role->syncPermissions($permissionIds);
            }

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.roles.index')
            ->with('status', "Imported {$imported} roles. Skipped {$skipped} rows.");
    }

    private function parsePermissionNames($value): array
    {
        if (! $value) {
            return [];
        }

        $items = preg_split('/[|,\n]/', (string) $value);

        return collect($items)
            ->map(fn ($item) => trim((string) $item))
            ->filter()
            ->unique()
            ->values()
            ->all();
    }

    private function filterPermissionsByGuard(array $permissionIds, string $guard): array
    {
        if (! $permissionIds) {
            return [];
        }

        return Permission::query()
            ->where('guard_name', $guard)
            ->whereIn('id', $permissionIds)
            ->pluck('id')
            ->all();
    }

    private function groupPermissions($permissions): array
    {
        return $permissions
            ->groupBy(function ($permission) {
                $name = $permission->name;
                if (str_contains($name, '.')) {
                    return Str::before($name, '.');
                }
                if (str_contains($name, ':')) {
                    return Str::before($name, ':');
                }

                return 'general';
            })
            ->sortKeys()
            ->all();
    }

    private function availableGuards(): array
    {
        $guards = array_keys(config('auth.guards', []));

        return $guards ?: ['web'];
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }
}
