<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportSecurityAuditLogsRequest;
use App\Http\Requests\Admin\StoreSecurityAuditLogRequest;
use App\Http\Requests\Admin\UpdateSecurityAuditLogRequest;
use App\Models\SecurityAuditLog;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;

class SecurityAuditLogController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $eventType = $request->query('event_type');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = SecurityAuditLog::query()->with('user');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('ip_address', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('event_type', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('status', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhereHas('user', function ($userQuery) use ($search) {
                        $userQuery->where('email', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('name', 'like', "%" . $this->escapeLike($search) . "%");
                    });
            });
        }

        if ($eventType) {
            $query->where('event_type', $eventType);
        }

        if ($status) {
            $query->where('status', $status);
        }

        $this->applySort($query, $sort);

        return view('admin.security-audit-logs.index', [
            'logs' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'eventType' => $eventType,
            'status' => $status,
            'sort' => $sort,
            'eventTypes' => config('security.audit_event_types', []),
            'statuses' => config('security.audit_statuses', []),
        ]);
    }

    public function create()
    {
        return view('admin.security-audit-logs.create', [
            'eventTypes' => config('security.audit_event_types', []),
            'statuses' => config('security.audit_statuses', []),
            'users' => User::query()->orderBy('name')->get(),
        ]);
    }

    public function store(StoreSecurityAuditLogRequest $request)
    {
        $data = $request->validated();

        SecurityAuditLog::create([
            'user_id' => $data['user_id'] ?? null,
            'event_type' => $data['event_type'],
            'status' => $data['status'],
            'ip_address' => $data['ip_address'] ?? null,
            'user_agent' => $data['user_agent'] ?? null,
            'metadata' => $this->parseMetadata($data['metadata'] ?? null),
            'logged_at' => $this->parseDateTime($data['logged_at'] ?? null) ?? now(),
        ]);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.security-audit-logs.create')
                ->with('status', 'Security audit log created. You can add another.');
        }

        return redirect()
            ->route('admin.security-audit-logs.index')
            ->with('status', 'Security audit log created successfully.');
    }

    public function show(SecurityAuditLog $securityAuditLog)
    {
        $securityAuditLog->load('user');

        return view('admin.security-audit-logs.show', [
            'log' => $securityAuditLog,
            'eventTypes' => config('security.audit_event_types', []),
            'statuses' => config('security.audit_statuses', []),
        ]);
    }

    public function edit(SecurityAuditLog $securityAuditLog)
    {
        return view('admin.security-audit-logs.edit', [
            'log' => $securityAuditLog,
            'eventTypes' => config('security.audit_event_types', []),
            'statuses' => config('security.audit_statuses', []),
            'users' => User::query()->orderBy('name')->get(),
        ]);
    }

    public function update(UpdateSecurityAuditLogRequest $request, SecurityAuditLog $securityAuditLog)
    {
        $data = $request->validated();

        $securityAuditLog->update([
            'user_id' => $data['user_id'] ?? null,
            'event_type' => $data['event_type'],
            'status' => $data['status'],
            'ip_address' => $data['ip_address'] ?? null,
            'user_agent' => $data['user_agent'] ?? null,
            'metadata' => $this->parseMetadata($data['metadata'] ?? null),
            'logged_at' => $this->parseDateTime($data['logged_at'] ?? null),
        ]);

        return redirect()
            ->route('admin.security-audit-logs.edit', $securityAuditLog)
            ->with('status', 'Security audit log updated successfully.');
    }

    public function destroy(SecurityAuditLog $securityAuditLog)
    {
        $securityAuditLog->delete();

        return redirect()
            ->route('admin.security-audit-logs.index')
            ->with('status', 'Security audit log deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:security_audit_logs,id'],
        ]);

        SecurityAuditLog::query()->whereIn('id', $validated['ids'])->delete();

        return redirect()
            ->route('admin.security-audit-logs.index')
            ->with('status', 'Selected security audit logs deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $eventType = $request->query('event_type');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = SecurityAuditLog::query()->with('user');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('ip_address', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('event_type', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('status', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhereHas('user', function ($userQuery) use ($search) {
                        $userQuery->where('email', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('name', 'like', "%" . $this->escapeLike($search) . "%");
                    });
            });
        }

        if ($eventType) {
            $query->where('event_type', $eventType);
        }

        if ($status) {
            $query->where('status', $status);
        }

        $this->applySort($query, $sort);

        $filename = 'security-audit-logs-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['event_type', 'status', 'user_email', 'ip_address', 'user_agent', 'metadata', 'logged_at']);

            $query->chunk(200, function ($logs) use ($handle) {
                foreach ($logs as $log) {
                    fputcsv($handle, [
                        $log->event_type,
                        $log->status,
                        $log->user?->email,
                        $log->ip_address,
                        $log->user_agent,
                        $log->metadata ? json_encode($log->metadata) : '',
                        optional($log->logged_at)->format('Y-m-d H:i:s'),
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportSecurityAuditLogsRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.security-audit-logs.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.security-audit-logs.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, ['event_type', 'status', 'user_email', 'ip_address', 'user_agent', 'metadata', 'logged_at']);
        $eventTypes = array_keys(config('security.audit_event_types', []));
        $statuses = array_keys(config('security.audit_statuses', []));

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $eventType = $this->normalizeValue($data['event_type'] ?? '', $eventTypes);
            $status = $this->normalizeValue($data['status'] ?? '', $statuses);

            if (! $eventType || ! $status) {
                $skipped++;
                continue;
            }

            $userId = null;
            if (! empty($data['user_email'])) {
                $userId = User::query()->where('email', trim((string) $data['user_email']))->value('id');
            }

            SecurityAuditLog::create([
                'user_id' => $userId,
                'event_type' => $eventType,
                'status' => $status,
                'ip_address' => $data['ip_address'] ?? null,
                'user_agent' => $data['user_agent'] ?? null,
                'metadata' => $this->parseMetadata($data['metadata'] ?? null),
                'logged_at' => $this->parseDateTime($data['logged_at'] ?? null) ?? now(),
            ]);

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.security-audit-logs.index')
            ->with('status', "Imported {$imported} security audit logs. Skipped {$skipped} rows.");
    }

    private function parseDateTime(?string $value): ?Carbon
    {
        if (! $value) {
            return null;
        }

        try {
            return Carbon::parse($value);
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function parseMetadata(?string $value): ?array
    {
        if (! $value) {
            return null;
        }

        $value = trim($value);
        if ($value === '') {
            return null;
        }

        $decoded = json_decode($value, true);
        if (json_last_error() === JSON_ERROR_NONE) {
            return $decoded;
        }

        return ['note' => $value];
    }

    private function normalizeValue(?string $value, array $allowed): ?string
    {
        $value = Str::lower(trim((string) $value));

        return in_array($value, $allowed, true) ? $value : null;
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'event_type_asc' => $query->orderBy('event_type'),
            'event_type_desc' => $query->orderBy('event_type', 'desc'),
            'oldest' => $query->orderBy('logged_at', 'asc'),
            default => $query->orderBy('logged_at', 'desc'),
        };
    }
}
