<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\UpdateSecuritySettingsRequest;
use App\Models\SystemSetting;
use App\Support\SecurityAuditLogger;
use App\Support\SecuritySettings;
use App\Support\TwoFactorService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class SecuritySettingsController extends Controller
{
    public function index(Request $request)
    {
        $setting = SystemSetting::query()->first();
        $payload = $setting?->payload ?? [];
        $defaults = config('security.defaults', []);

        $settings = array_merge($defaults, $payload['security'] ?? []);
        $user = $request->user();

        $twoFactorSecret = null;
        $otpAuthUrl = null;

        if ($user && ! $user->two_factor_enabled) {
            $twoFactorSecret = $request->session()->get('two_factor_settings_secret');
            if (! $twoFactorSecret) {
                $twoFactorSecret = TwoFactorService::generateSecret();
                $request->session()->put('two_factor_settings_secret', $twoFactorSecret);
            }

            $issuer = $settings['two_factor_issuer'] ?? config('app.name', 'SnapNest');
            $otpAuthUrl = TwoFactorService::buildOtpAuthUrl($user, $twoFactorSecret, $issuer);
        }

        return view('admin.security-settings.index', [
            'settings' => $settings,
            'adminRoles' => config('security.admin_roles', []),
            'twoFactorEnabled' => $user?->two_factor_enabled ?? false,
            'twoFactorSecret' => $twoFactorSecret,
            'twoFactorOtpAuthUrl' => $otpAuthUrl,
        ]);
    }

    public function update(UpdateSecuritySettingsRequest $request)
    {
        $setting = SystemSetting::query()->firstOrCreate([]);
        $payload = $setting->payload ?? [];
        $data = $request->validated();

        $data['captcha_enabled'] = $request->boolean('captcha_enabled');
        $data['two_factor_admin_only'] = $request->boolean('two_factor_admin_only');

        $payload['security'] = array_merge($payload['security'] ?? [], $data);
        $setting->payload = $payload;
        $setting->updated_by = $request->user()?->id;
        $setting->save();

        return back()->with('status', 'Security settings updated successfully.');
    }

    public function enableTwoFactor(Request $request)
    {
        $request->validate([
            'code' => ['required', 'string'],
        ]);

        $user = $request->user();

        if (! $user || $user->two_factor_enabled) {
            return back()->with('status', 'Two-factor authentication is already enabled.');
        }

        $secret = $request->session()->get('two_factor_settings_secret');

        if (! $secret) {
            return back()->with('error', 'Two-factor setup expired. Refresh the page to generate a new secret.');
        }

        if (! TwoFactorService::verifyCode($secret, $request->input('code'))) {
            SecurityAuditLogger::log('two_factor_failed', 'failed', $user, $request);

            return back()->withErrors([
                'code' => 'The authentication code is invalid.',
            ]);
        }

        $user->forceFill([
            'two_factor_enabled' => true,
            'two_factor_secret' => $secret,
            'two_factor_confirmed_at' => now(),
        ])->save();

        $request->session()->forget('two_factor_settings_secret');

        SecurityAuditLogger::log('two_factor_enabled', 'success', $user, $request);

        return back()->with('status', 'Two-factor authentication enabled.');
    }

    public function disableTwoFactor(Request $request)
    {
        $request->validate([
            'current_password' => ['required', 'string'],
        ]);

        $user = $request->user();

        if (! $user || ! $user->two_factor_enabled) {
            return back()->with('status', 'Two-factor authentication is already disabled.');
        }

        if (! Hash::check($request->input('current_password'), $user->password)) {
            return back()->withErrors([
                'current_password' => 'The provided password is incorrect.',
            ]);
        }

        $user->forceFill([
            'two_factor_enabled' => false,
            'two_factor_secret' => null,
            'two_factor_confirmed_at' => null,
            'two_factor_recovery_codes' => null,
        ])->save();

        SecurityAuditLogger::log('two_factor_disabled', 'info', $user, $request);

        return back()->with('status', 'Two-factor authentication disabled.');
    }
}
