<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportStatesRequest;
use App\Http\Requests\Admin\StoreStateRequest;
use App\Http\Requests\Admin\UpdateStateRequest;
use App\Models\Country;
use App\Models\State;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class StateController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $countryFilter = $request->query('country');
        $sort = $request->query('sort', 'name_asc');

        $query = State::query()->with(['country'])->withCount('cities');

        if ($search) {
            $query->where('name', 'like', "%" . $this->escapeLike($search) . "%");
        }

        if ($countryFilter) {
            $query->where('country_id', $countryFilter);
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return view('admin.locations.states.index', [
            'states' => $query->paginate(10)->withQueryString(),
            'countries' => Country::query()->orderBy('name')->get(),
            'search' => $search,
            'status' => $status,
            'countryFilter' => $countryFilter,
            'sort' => $sort,
        ]);
    }

    public function create()
    {
        return view('admin.locations.states.create', [
            'countries' => Country::query()->orderBy('name')->get(),
        ]);
    }

    public function store(StoreStateRequest $request)
    {
        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        State::create($data);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.states.create')
                ->with('status', 'State created. You can add another.');
        }

        return redirect()
            ->route('admin.states.index')
            ->with('status', 'State created successfully.');
    }

    public function show(State $state)
    {
        $state->load(['country', 'cities' => function ($query) {
            $query->orderBy('name');
        }]);

        return view('admin.locations.states.show', [
            'state' => $state,
        ]);
    }

    public function edit(State $state)
    {
        return view('admin.locations.states.edit', [
            'state' => $state,
            'countries' => Country::query()->orderBy('name')->get(),
        ]);
    }

    public function update(UpdateStateRequest $request, State $state)
    {
        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        $state->update($data);

        return redirect()
            ->route('admin.states.edit', $state)
            ->with('status', 'State updated successfully.');
    }

    public function destroy(State $state)
    {
        $state->delete();

        return redirect()
            ->route('admin.states.index')
            ->with('status', 'State deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:states,id'],
        ]);

        State::query()->whereIn('id', $validated['ids'])->delete();

        return redirect()
            ->route('admin.states.index')
            ->with('status', 'Selected states deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $countryFilter = $request->query('country');
        $sort = $request->query('sort', 'name_asc');

        $query = State::query()->with(['country'])->withCount('cities');

        if ($search) {
            $query->where('name', 'like', "%" . $this->escapeLike($search) . "%");
        }

        if ($countryFilter) {
            $query->where('country_id', $countryFilter);
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $filename = 'states-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['country', 'country_iso2', 'state', 'code', 'cities_count', 'is_active']);

            $query->chunk(200, function ($states) use ($handle) {
                foreach ($states as $state) {
                    fputcsv($handle, [
                        $state->country?->name,
                        $state->country?->iso2,
                        $state->name,
                        $state->code,
                        $state->cities_count,
                        $state->is_active ? 1 : 0,
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportStatesRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.states.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.states.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, ['country', 'country_iso2', 'state', 'name', 'code', 'is_active']);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $countryId = $this->resolveCountryId($data);
            $stateName = trim((string) ($data['state'] ?? $data['name'] ?? ''));

            if (! $countryId || $stateName === '') {
                $skipped++;
                continue;
            }

            State::updateOrCreate(
                [
                    'country_id' => $countryId,
                    'name' => $stateName,
                ],
                [
                    'code' => trim((string) ($data['code'] ?? '')) ?: null,
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.states.index')
            ->with('status', "Imported {$imported} states. Skipped {$skipped} rows.");
    }

    private function resolveCountryId(array $data): ?int
    {
        $countryIso2 = trim((string) ($data['country_iso2'] ?? ''));
        if ($countryIso2 !== '') {
            return Country::query()
                ->where('iso2', strtoupper($countryIso2))
                ->value('id');
        }

        $countryName = trim((string) ($data['country'] ?? ''));
        if ($countryName !== '') {
            return Country::query()
                ->where('name', $countryName)
                ->value('id');
        }

        return null;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_desc' => $query->orderBy('name', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('name'),
        };
    }
}
