<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportSystemLogsRequest;
use App\Http\Requests\Admin\StoreSystemLogRequest;
use App\Http\Requests\Admin\UpdateSystemLogRequest;
use App\Models\SystemLog;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;

class SystemLogController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $level = $request->query('level');
        $sort = $request->query('sort', 'latest');

        $query = SystemLog::query()->with('createdBy');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('message', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('source', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($level) {
            $query->where('level', $level);
        }

        $this->applySort($query, $sort);

        return view('admin.system-logs.index', [
            'logs' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'level' => $level,
            'sort' => $sort,
            'levels' => config('system.log_levels', []),
        ]);
    }

    public function create()
    {
        return view('admin.system-logs.create', [
            'levels' => config('system.log_levels', []),
        ]);
    }

    public function store(StoreSystemLogRequest $request)
    {
        $data = $request->validated();

        SystemLog::create([
            'level' => $data['level'],
            'source' => $data['source'] ?? null,
            'message' => $data['message'],
            'context' => $this->parseContext($data['context'] ?? null),
            'logged_at' => $this->parseDateTime($data['logged_at'] ?? null) ?? now(),
            'created_by' => $request->user()?->id,
        ]);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.system-logs.create')
                ->with('status', 'System log created. You can add another.');
        }

        return redirect()
            ->route('admin.system-logs.index')
            ->with('status', 'System log created successfully.');
    }

    public function show(SystemLog $systemLog)
    {
        $systemLog->load('createdBy');

        return view('admin.system-logs.show', [
            'log' => $systemLog,
            'levels' => config('system.log_levels', []),
        ]);
    }

    public function edit(SystemLog $systemLog)
    {
        return view('admin.system-logs.edit', [
            'log' => $systemLog,
            'levels' => config('system.log_levels', []),
        ]);
    }

    public function update(UpdateSystemLogRequest $request, SystemLog $systemLog)
    {
        $data = $request->validated();

        $systemLog->update([
            'level' => $data['level'],
            'source' => $data['source'] ?? null,
            'message' => $data['message'],
            'context' => $this->parseContext($data['context'] ?? null),
            'logged_at' => $this->parseDateTime($data['logged_at'] ?? null),
        ]);

        return redirect()
            ->route('admin.system-logs.edit', $systemLog)
            ->with('status', 'System log updated successfully.');
    }

    public function destroy(SystemLog $systemLog)
    {
        $systemLog->delete();

        return redirect()
            ->route('admin.system-logs.index')
            ->with('status', 'System log deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:system_logs,id'],
        ]);

        SystemLog::query()->whereIn('id', $validated['ids'])->delete();

        return redirect()
            ->route('admin.system-logs.index')
            ->with('status', 'Selected system logs deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $level = $request->query('level');
        $sort = $request->query('sort', 'latest');

        $query = SystemLog::query()->with('createdBy');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('message', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('source', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($level) {
            $query->where('level', $level);
        }

        $this->applySort($query, $sort);

        $filename = 'system-logs-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['level', 'source', 'message', 'context', 'logged_at', 'created_by']);

            $query->chunk(200, function ($logs) use ($handle) {
                foreach ($logs as $log) {
                    fputcsv($handle, [
                        $log->level,
                        $log->source,
                        $log->message,
                        $log->context ? json_encode($log->context) : '',
                        optional($log->logged_at)->format('Y-m-d H:i:s'),
                        $log->createdBy?->email,
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportSystemLogsRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.system-logs.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.system-logs.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, ['level', 'source', 'message', 'context', 'logged_at']);
        $levels = array_keys(config('system.log_levels', []));

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $message = trim((string) ($data['message'] ?? ''));

            if ($message === '') {
                $skipped++;
                continue;
            }

            $level = $this->normalizeLevel($data['level'] ?? null, $levels);

            SystemLog::create([
                'level' => $level,
                'source' => $data['source'] ?? null,
                'message' => $message,
                'context' => $this->parseContext($data['context'] ?? null),
                'logged_at' => $this->parseDateTime($data['logged_at'] ?? null) ?? now(),
            ]);

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.system-logs.index')
            ->with('status', "Imported {$imported} system logs. Skipped {$skipped} rows.");
    }

    private function parseDateTime(?string $value): ?Carbon
    {
        if (! $value) {
            return null;
        }

        try {
            return Carbon::parse($value);
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function parseContext(?string $value): ?array
    {
        if (! $value) {
            return null;
        }

        $value = trim($value);
        if ($value === '') {
            return null;
        }

        $decoded = json_decode($value, true);
        if (json_last_error() === JSON_ERROR_NONE) {
            return $decoded;
        }

        return ['note' => $value];
    }

    private function normalizeLevel(?string $value, array $levels): string
    {
        $value = Str::lower(trim((string) $value));

        return in_array($value, $levels, true) ? $value : 'info';
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'oldest' => $query->orderBy('logged_at', 'asc'),
            'level_asc' => $query->orderBy('level'),
            'level_desc' => $query->orderBy('level', 'desc'),
            default => $query->orderBy('logged_at', 'desc'),
        };
    }
}
