<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\UpdateSystemSettingsRequest;
use App\Models\City;
use App\Models\Country;
use App\Models\Currency;
use App\Models\Language;
use App\Models\State;
use App\Models\SystemSetting;
use App\Jobs\MigrateClientStorageMedia;
use App\Services\CloudStorageManager;
use App\Services\StorageUsageService;
use App\Support\CloudStorageSettings;
use App\Support\FormatSettings;
use App\Support\SeoFileManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;

class SystemSettingsController extends Controller
{
    public function index()
    {
        $setting = SystemSetting::query()->first();
        $payload = $setting?->payload ?? [];

        $socialProviders = config('system.social_providers', []);
        $defaultProviders = [];
        foreach ($socialProviders as $key => $label) {
            $defaultProviders[$key] = [
                'enabled' => false,
                'client_id' => '',
                'client_secret' => '',
                'redirect_url' => '',
            ];
        }

        $defaults = [
            'general' => [
                'default_language' => '',
                'default_currency' => '',
                'default_timezone' => '',
                'date_format' => '',
                'time_format' => '',
                'support_email' => '',
                'support_phone' => '',
                'enable_multi_language' => false,
            ],
            'site' => [
                'site_name' => '',
                'site_logo' => null,
                'favicon' => null,
                'admin_logo' => null,
                'company_name' => '',
                'company_address' => '',
                'company_country_id' => null,
                'company_state_id' => null,
                'company_city_id' => null,
                'company_phone' => '',
                'business_hours' => '',
                'latitude' => '',
                'longitude' => '',
                'right_click_disable' => false,
                'uc_browser_block' => false,
            ],
            'smtp' => [
                'smtp_host' => '',
                'smtp_port' => '',
                'smtp_username' => '',
                'smtp_password' => '',
                'smtp_encryption' => 'tls',
                'smtp_from_address' => '',
                'smtp_from_name' => '',
            ],
            'footer' => [
                'footer_logo' => null,
                'footer_social' => [
                    'facebook' => '',
                    'x' => '',
                    'linkedin' => '',
                    'instagram' => '',
                    'whatsapp' => '',
                ],
                'app_links' => [
                    'android' => '',
                    'ios' => '',
                    'direct' => '',
                ],
            ],
            'seo' => [
                'meta_title' => '',
                'meta_description' => '',
                'meta_keywords' => '',
                'sitemap_enabled' => false,
                'robots_txt' => '',
                'llm_txt' => '',
            ],
            'social' => [
                'providers' => $defaultProviders,
            ],
            'gdpr' => [
                'gdpr_enabled' => false,
                'gdpr_message' => '',
                'gdpr_accept_label' => 'Accept',
                'gdpr_decline_label' => 'Decline',
            ],
            'media' => [
                'max_file_size' => (int) config('events.media.max_file_size', 51200),
                'chunk_size' => (int) config('events.media.chunk_size', 5242880),
                'max_width' => (int) config('events.media.max_width', 2000),
                'quality' => (int) config('events.media.quality', 82),
                'storage_limit_gb' => (float) config('events.media.storage_limit_gb', 6.1),
                'watermark' => [
                    'enabled' => (bool) config('events.media.watermark.enabled', true),
                    'image' => (string) config('events.media.watermark.image', ''),
                    'position' => (string) config('events.media.watermark.position', 'top_right'),
                ],
                'ftp_path' => (string) config('events.media.ftp_path', storage_path('app/ftp')),
            ],
            'cloud_storage' => CloudStorageSettings::get(),
            'maintenance' => [
                'maintenance_enabled' => false,
                'maintenance_message' => '',
            ],
            'pwa' => [
                'enabled'          => false,
                'app_name'         => '',
                'app_url'          => '/',
                'icon'             => null,
                'splash_screen'    => null,
                'theme_color'      => '#156055',
                'background_color' => '#f5f0e8',
            ],
        ];

        $settings = array_replace_recursive($defaults, $payload);

        $storageSummary = app(StorageUsageService::class)->summary();

        return view('admin.system-settings.index', [
            'settings' => $settings,
            'socialProviders' => $socialProviders,
            'countries' => Country::query()->orderBy('name')->get(),
            'states' => State::query()->orderBy('name')->get(),
            'cities' => City::query()->orderBy('name')->get(),
            'currencies' => Currency::query()->orderBy('name')->get(),
            'languages' => Language::query()->orderBy('name')->get(),
            'storageSummary' => $storageSummary,
        ]);
    }

    public function update(UpdateSystemSettingsRequest $request, string $group)
    {
        // Enforce plan feature: Cloud storage (Admin/Super Admin bypass automatically)
        if ($group === 'cloud_storage') {
            $planLimit = app(\App\Services\PlanLimitService::class);
            if (! $planLimit->hasFeature($request->user(), 'has_cloud_storage')) {
                return back()->with('error', 'Cloud storage is not available on your current plan. Please upgrade.');
            }
        }

        $setting = SystemSetting::query()->firstOrCreate([]);
        $payload = $setting->payload ?? [];
        $data = $request->validated();

        switch ($group) {
            case 'general':
                $data['enable_multi_language'] = $request->boolean('enable_multi_language');
                break;
            case 'site':
                $data['right_click_disable'] = $request->boolean('right_click_disable');
                $data['uc_browser_block'] = $request->boolean('uc_browser_block');
                $data = $this->handleUpload($request, $data, 'site_logo', 'settings/site');
                $data = $this->handleUpload($request, $data, 'favicon', 'settings/site');
                $data = $this->handleUpload($request, $data, 'admin_logo', 'settings/site');
                break;
            case 'smtp':
                break;
            case 'footer':
                $data = $this->handleUpload($request, $data, 'footer_logo', 'settings/footer');
                $data['footer_social'] = $this->normalizeLinks(
                    $data['footer_social'] ?? [],
                    ['facebook', 'x', 'linkedin', 'instagram', 'whatsapp']
                );
                $data['app_links'] = $this->normalizeLinks(
                    $data['app_links'] ?? [],
                    ['android', 'ios', 'direct']
                );
                break;
            case 'seo':
                $data['sitemap_enabled'] = $request->boolean('sitemap_enabled');
                break;
            case 'social':
                $data['providers'] = $this->normalizeProviders($data['providers'] ?? []);
                break;
            case 'gdpr':
                $data['gdpr_enabled'] = $request->boolean('gdpr_enabled');
                break;
            case 'media':
                $maxFileSize = $data['media_max_file_size'] ?? $data['max_file_size'] ?? null;
                $data['max_file_size'] = $this->convertMbToKb($maxFileSize);
                $data['chunk_size'] = $this->convertMbToBytes($data['chunk_size'] ?? null);
                $data['max_width'] = isset($data['max_width']) ? (int) $data['max_width'] : null;
                $data['quality'] = isset($data['quality']) ? (int) $data['quality'] : null;
                $data['storage_limit_gb'] = isset($data['storage_limit_gb']) ? (float) $data['storage_limit_gb'] : null;
                $data['ftp_path'] = isset($data['ftp_path']) ? trim((string) $data['ftp_path']) : null;
                $data = $this->handleUpload($request, $data, 'watermark_image', 'settings/media');
                $watermark = is_array($payload['media']['watermark'] ?? null) ? $payload['media']['watermark'] : [];
                $watermarkImage = $data['watermark_image'] ?? ($watermark['image'] ?? null);
                $watermarkPosition = trim((string) ($data['watermark_position'] ?? ''));
                if ($watermarkPosition === '') {
                    $watermarkPosition = (string) config('events.media.watermark.position', 'top_right');
                }
                $data['watermark'] = [
                    'enabled' => $request->boolean('watermark_enabled'),
                    'image' => $watermarkImage,
                    'position' => $watermarkPosition,
                ];
                unset($data['media_max_file_size'], $data['watermark_enabled'], $data['watermark_image'], $data['watermark_position']);
                break;
            case 'maintenance':
                $data['maintenance_enabled'] = $request->boolean('maintenance_enabled');
                break;
            case 'pwa':
                $data['enabled'] = $request->boolean('pwa_enabled');
                $data = $this->handleUpload($request, $data, 'icon', 'settings/pwa');
                $data = $this->handleUpload($request, $data, 'splash_screen', 'settings/pwa');
                unset($data['pwa_enabled']);
                break;
            case 'cloud_storage':
                $data['enabled'] = $request->boolean('enabled');
                $data['visibility'] = trim((string) ($data['visibility'] ?? 'private')) ?: 'private';
                $data['provider'] = trim((string) ($data['provider'] ?? 's3')) ?: 's3';
                $data['base_path'] = trim((string) ($data['base_path'] ?? ''));
                $data['limits'] = [
                    'enabled' => $request->boolean('limit_enabled'),
                    'value' => isset($data['limit_value']) ? (float) $data['limit_value'] : null,
                    'unit' => trim((string) ($data['limit_unit'] ?? 'gb')) ?: 'gb',
                    'configured' => true,
                ];
                $data['providers'] = $this->normalizeCloudProviders($data['providers'] ?? [], $payload['cloud_storage']['providers'] ?? []);
                unset($data['limit_enabled'], $data['limit_value'], $data['limit_unit']);
                break;
        }

        $payload[$group] = array_merge($payload[$group] ?? [], $data);
        $setting->payload = $payload;
        $setting->updated_by = $request->user()?->id;
        $setting->save();

        if ($group === 'general') {
            FormatSettings::reset();
        }
        if ($group === 'seo') {
            SeoFileManager::sync($payload['seo'] ?? []);
        }
        if ($group === 'media') {
            \App\Support\EventMediaSettings::forgetCache();
        }
        if ($group === 'cloud_storage') {
            CloudStorageSettings::forgetCache();
        }

        return back()->with('status', 'Settings updated successfully.');
    }

    public function testCloudStorage(Request $request, CloudStorageManager $manager)
    {
        $planLimit = app(\App\Services\PlanLimitService::class);
        if (! $planLimit->hasFeature($request->user(), 'has_cloud_storage')) {
            return back()->with('error', 'Cloud storage is not available on your current plan.');
        }

        $result = $manager->testConnection();
        $setting = SystemSetting::query()->firstOrCreate([]);
        $payload = $setting->payload ?? [];
        $cloud = is_array($payload['cloud_storage'] ?? null) ? $payload['cloud_storage'] : [];

        $cloud['last_test_at'] = now()->toDateTimeString();
        $cloud['last_test_status'] = $result['ok'] ? 'connected' : 'error';
        $cloud['last_test_message'] = $result['message'];

        $payload['cloud_storage'] = $cloud;
        $setting->payload = $payload;
        $setting->updated_by = $request->user()?->id;
        $setting->save();

        return back()->with($result['ok'] ? 'status' : 'error', $result['message']);
    }

    public function startCloudStorageMigration(Request $request, CloudStorageManager $manager)
    {
        $planLimit = app(\App\Services\PlanLimitService::class);
        if (! $planLimit->hasFeature($request->user(), 'has_cloud_storage')) {
            return back()->with('error', 'Cloud storage is not available on your current plan.');
        }

        if (! $manager->hasProviderConfig()) {
            return back()->with('error', 'Cloud storage is not configured yet.');
        }

        $setting = SystemSetting::query()->firstOrCreate([]);
        $payload = $setting->payload ?? [];
        $cloud = is_array($payload['cloud_storage'] ?? null) ? $payload['cloud_storage'] : [];
        $migration = is_array($cloud['migration'] ?? null) ? $cloud['migration'] : [];

        if (($migration['status'] ?? null) === 'running') {
            return back()->with('status', 'Migration is already running.');
        }

        $cloud['migration'] = [
            'status' => 'running',
            'processed' => 0,
            'failed' => 0,
            'total' => \App\Models\EventMedia::query()->count(),
            'started_at' => now()->toDateTimeString(),
            'finished_at' => null,
            'last_error' => '',
            'last_id' => null,
        ];

        $payload['cloud_storage'] = $cloud;
        $setting->payload = $payload;
        $setting->updated_by = $request->user()?->id;
        $setting->save();

        MigrateClientStorageMedia::dispatch();

        return back()->with('status', 'Migration started. This runs in the background.');
    }

    public function testSmtp(Request $request)
    {
        $data = $request->validate([
            'test_email' => ['required', 'email', 'max:255'],
        ]);

        $recipient = $data['test_email'];
        $appName = config('app.name', 'SnapNest – Event Photo Sharing Platform with QR & Face Recognition');

        try {
            Mail::raw("SMTP test email from {$appName}.", function ($message) use ($recipient, $appName) {
                $message->to($recipient)
                    ->subject("{$appName} SMTP Test");
            });
        } catch (\Throwable $exception) {
            return back()->with('error', 'SMTP test failed: ' . $exception->getMessage());
        }

        return back()->with('status', 'SMTP test email sent to ' . $recipient . '.');
    }

    private function handleUpload(Request $request, array $data, string $field, string $path): array
    {
        if ($request->hasFile($field)) {
            $data[$field] = $request->file($field)->store($path, 'public');
        }

        return $data;
    }

    private function normalizeLinks(array $links, array $keys): array
    {
        $normalized = [];

        foreach ($keys as $key) {
            $normalized[$key] = isset($links[$key]) ? trim((string) $links[$key]) : '';
        }

        return $normalized;
    }

    private function normalizeProviders(array $providers): array
    {
        $defaults = [];
        $available = config('system.social_providers', []);

        foreach ($available as $key => $label) {
            $data = $providers[$key] ?? [];
            $defaults[$key] = [
                'enabled' => filter_var($data['enabled'] ?? false, FILTER_VALIDATE_BOOLEAN),
                'client_id' => trim((string) ($data['client_id'] ?? '')),
                'client_secret' => trim((string) ($data['client_secret'] ?? '')),
                'redirect_url' => trim((string) ($data['redirect_url'] ?? '')),
            ];
        }

        return $defaults;
    }

    private function normalizeCloudProviders(array $providers, array $existing): array
    {
        $normalized = $existing;

        $normalized['s3'] = $this->mergeProviderCredentials(
            $existing['s3'] ?? [],
            $providers['s3'] ?? [],
            ['key', 'secret']
        );

        $normalized['gcs'] = $this->mergeProviderCredentials(
            $existing['gcs'] ?? [],
            $providers['gcs'] ?? [],
            ['key_file']
        );

        $normalized['gdrive'] = $this->mergeProviderCredentials(
            $existing['gdrive'] ?? [],
            $providers['gdrive'] ?? [],
            ['client_secret', 'refresh_token']
        );

        $normalized['azure'] = $this->mergeProviderCredentials(
            $existing['azure'] ?? [],
            $providers['azure'] ?? [],
            ['account_key']
        );

        $normalized['bunny'] = $this->mergeProviderCredentials(
            $existing['bunny'] ?? [],
            $providers['bunny'] ?? [],
            ['access_key']
        );

        $normalized['wasabi'] = $this->mergeProviderCredentials(
            $existing['wasabi'] ?? [],
            $providers['wasabi'] ?? [],
            ['key', 'secret']
        );

        return array_replace_recursive($existing, $normalized);
    }

    private function mergeProviderCredentials(array $existing, array $incoming, array $keys): array
    {
        $merged = array_merge($existing, $incoming);

        foreach ($keys as $key) {
            $value = trim((string) ($incoming[$key] ?? ''));
            if ($value !== '') {
                $merged[$key] = CloudStorageSettings::encryptValue($value);
            } else {
                $merged[$key] = $existing[$key] ?? '';
            }
        }

        return $merged;
    }

    private function convertMbToKb($value): ?int
    {
        if ($value === null || trim((string) $value) === '') {
            return null;
        }

        return (int) round(((float) $value) * 1024);
    }

    private function convertMbToBytes($value): ?int
    {
        if ($value === null || trim((string) $value) === '') {
            return null;
        }

        return (int) round(((float) $value) * 1024 * 1024);
    }
}
