<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportTestimonialsRequest;
use App\Http\Requests\Admin\StoreTestimonialRequest;
use App\Http\Requests\Admin\UpdateTestimonialRequest;
use App\Models\Testimonial;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class TestimonialController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = Testimonial::query()->with('creator');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('client_name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('client_position', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('review', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return view('admin.content.testimonials.index', [
            'testimonials' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'status' => $status,
            'sort' => $sort,
        ]);
    }

    public function create()
    {
        return view('admin.content.testimonials.create');
    }

    public function store(StoreTestimonialRequest $request)
    {
        $data = $request->validated();

        if ($request->hasFile('client_image')) {
            $data['client_image'] = $request->file('client_image')->store('content/testimonials', 'public');
        }

        $data['is_active'] = $request->boolean('is_active');
        $data['created_by'] = $request->user()?->id;

        Testimonial::create($data);

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.testimonials.create')
                ->with('status', 'Testimonial created. You can add another.');
        }

        return redirect()
            ->route('admin.testimonials.index')
            ->with('status', 'Testimonial created successfully.');
    }

    public function show(Testimonial $testimonial)
    {
        $testimonial->load('creator');

        return view('admin.content.testimonials.show', [
            'testimonial' => $testimonial,
        ]);
    }

    public function edit(Testimonial $testimonial)
    {
        $testimonial->load('creator');

        return view('admin.content.testimonials.edit', [
            'testimonial' => $testimonial,
        ]);
    }

    public function update(UpdateTestimonialRequest $request, Testimonial $testimonial)
    {
        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        if ($request->hasFile('client_image')) {
            if ($testimonial->client_image) {
                Storage::disk('public')->delete($testimonial->client_image);
            }
            $data['client_image'] = $request->file('client_image')->store('content/testimonials', 'public');
        }

        $testimonial->update($data);

        return redirect()
            ->route('admin.testimonials.edit', $testimonial)
            ->with('status', 'Testimonial updated successfully.');
    }

    public function destroy(Testimonial $testimonial)
    {
        if ($testimonial->client_image) {
            Storage::disk('public')->delete($testimonial->client_image);
        }

        $testimonial->delete();

        return redirect()
            ->route('admin.testimonials.index')
            ->with('status', 'Testimonial deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:testimonials,id'],
        ]);

        $testimonials = Testimonial::query()->whereIn('id', $validated['ids'])->get();

        foreach ($testimonials as $testimonial) {
            if ($testimonial->client_image) {
                Storage::disk('public')->delete($testimonial->client_image);
            }
            $testimonial->delete();
        }

        return redirect()
            ->route('admin.testimonials.index')
            ->with('status', 'Selected testimonials deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = Testimonial::query()->with('creator');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('client_name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('client_position', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('review', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $filename = 'testimonials-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['client_name', 'client_position', 'rating', 'review', 'is_active', 'created_by', 'created_at']);

            $query->chunk(200, function ($testimonials) use ($handle) {
                foreach ($testimonials as $testimonial) {
                    fputcsv($handle, [
                        $testimonial->client_name,
                        $testimonial->client_position,
                        $testimonial->rating,
                        $testimonial->review,
                        $testimonial->is_active ? 1 : 0,
                        $testimonial->creator?->email,
                        optional($testimonial->created_at)->format('Y-m-d H:i:s'),
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportTestimonialsRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.testimonials.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.testimonials.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, ['client_name', 'client_position', 'rating', 'review', 'is_active']);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['client_name'] ?? ''));
            $review = trim((string) ($data['review'] ?? ''));

            if ($name === '' || $review === '') {
                $skipped++;
                continue;
            }

            $position = trim((string) ($data['client_position'] ?? ''));
            $rating = $this->normalizeRating($data['rating'] ?? null);

            Testimonial::updateOrCreate(
                [
                    'client_name' => $name,
                    'client_position' => $position !== '' ? $position : null,
                ],
                [
                    'review' => $review,
                    'rating' => $rating,
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.testimonials.index')
            ->with('status', "Imported {$imported} testimonials. Skipped {$skipped} rows.");
    }

    private function normalizeRating($value): int
    {
        $rating = (int) $value;

        if ($rating < 1) {
            return 1;
        }

        if ($rating > 5) {
            return 5;
        }

        return $rating ?: 5;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_asc' => $query->orderBy('client_name'),
            'name_desc' => $query->orderBy('client_name', 'desc'),
            'rating_asc' => $query->orderBy('rating'),
            'rating_desc' => $query->orderBy('rating', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }
}
