<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportTicketAssignmentsRequest;
use App\Http\Requests\Admin\StoreTicketAssignmentRequest;
use App\Http\Requests\Admin\UpdateTicketAssignmentRequest;
use App\Models\SupportTicket;
use App\Models\SupportTicketAssignment;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class TicketAssignmentController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $ticketFilter = $request->query('ticket');
        $assignee = $request->query('assignee');
        $sort = $request->query('sort', 'latest');

        $query = SupportTicketAssignment::query()
            ->with(['ticket', 'assignedTo', 'assignedBy']);

        if ($search) {
            $query->whereHas('ticket', function ($builder) use ($search) {
                $builder->where('ticket_number', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('subject', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        if ($ticketFilter) {
            $query->where('support_ticket_id', $ticketFilter);
        }

        if ($assignee) {
            $query->where('assigned_to', $assignee);
        }

        $this->applySort($query, $sort);

        return view('admin.ticket-assignments.index', [
            'assignments' => $query->paginate(10)->withQueryString(),
            'search' => $search,
            'status' => $status,
            'ticketFilter' => $ticketFilter,
            'assignee' => $assignee,
            'sort' => $sort,
            'tickets' => SupportTicket::query()->orderBy('created_at', 'desc')->limit(200)->get(),
            'agents' => User::role('Support Agent')->orderBy('name')->get(),
        ]);
    }

    public function create()
    {
        return view('admin.ticket-assignments.create', [
            'tickets' => SupportTicket::query()->orderBy('created_at', 'desc')->limit(200)->get(),
            'agents' => User::role('Support Agent')->orderBy('name')->get(),
        ]);
    }

    public function store(StoreTicketAssignmentRequest $request)
    {
        $data = $request->validated();

        DB::transaction(function () use ($data, $request) {
            if ($request->boolean('is_active')) {
                SupportTicketAssignment::query()
                    ->where('support_ticket_id', $data['support_ticket_id'])
                    ->update(['is_active' => false]);
            }

            SupportTicketAssignment::create([
                'support_ticket_id' => $data['support_ticket_id'],
                'assigned_to' => $data['assigned_to'],
                'assigned_by' => $request->user()?->id,
                'note' => $data['note'] ?? null,
                'is_active' => $request->boolean('is_active'),
                'assigned_at' => $data['assigned_at'] ?? now(),
            ]);
        });

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.ticket-assignments.create')
                ->with('status', 'Assignment created. You can add another.');
        }

        return redirect()
            ->route('admin.ticket-assignments.index')
            ->with('status', 'Assignment created successfully.');
    }

    public function show(SupportTicketAssignment $ticketAssignment)
    {
        $ticketAssignment->load(['ticket', 'assignedTo', 'assignedBy']);

        return view('admin.ticket-assignments.show', [
            'assignment' => $ticketAssignment,
        ]);
    }

    public function edit(SupportTicketAssignment $ticketAssignment)
    {
        return view('admin.ticket-assignments.edit', [
            'assignment' => $ticketAssignment->load('ticket'),
            'tickets' => SupportTicket::query()->orderBy('created_at', 'desc')->limit(200)->get(),
            'agents' => User::role('Support Agent')->orderBy('name')->get(),
        ]);
    }

    public function update(UpdateTicketAssignmentRequest $request, SupportTicketAssignment $ticketAssignment)
    {
        $data = $request->validated();

        DB::transaction(function () use ($data, $request, $ticketAssignment) {
            if ($request->boolean('is_active')) {
                SupportTicketAssignment::query()
                    ->where('support_ticket_id', $data['support_ticket_id'])
                    ->where('id', '!=', $ticketAssignment->id)
                    ->update(['is_active' => false]);
            }

            $ticketAssignment->update([
                'support_ticket_id' => $data['support_ticket_id'],
                'assigned_to' => $data['assigned_to'],
                'note' => $data['note'] ?? null,
                'is_active' => $request->boolean('is_active'),
                'assigned_at' => $data['assigned_at'] ?? $ticketAssignment->assigned_at,
            ]);
        });

        return redirect()
            ->route('admin.ticket-assignments.edit', $ticketAssignment)
            ->with('status', 'Assignment updated successfully.');
    }

    public function destroy(SupportTicketAssignment $ticketAssignment)
    {
        $ticketAssignment->delete();

        return redirect()
            ->route('admin.ticket-assignments.index')
            ->with('status', 'Assignment deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:support_ticket_assignments,id'],
        ]);

        SupportTicketAssignment::query()->whereIn('id', $validated['ids'])->delete();

        return redirect()
            ->route('admin.ticket-assignments.index')
            ->with('status', 'Selected assignments deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $status = $request->query('status');
        $ticketFilter = $request->query('ticket');
        $assignee = $request->query('assignee');
        $sort = $request->query('sort', 'latest');

        $query = SupportTicketAssignment::query()->with(['ticket', 'assignedTo']);

        if ($search) {
            $query->whereHas('ticket', function ($builder) use ($search) {
                $builder->where('ticket_number', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('subject', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        if ($ticketFilter) {
            $query->where('support_ticket_id', $ticketFilter);
        }

        if ($assignee) {
            $query->where('assigned_to', $assignee);
        }

        $this->applySort($query, $sort);

        $filename = 'ticket-assignments-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, [
                'ticket_number',
                'subject',
                'assigned_to',
                'assigned_to_email',
                'assigned_at',
                'is_active',
                'note',
            ]);

            $query->chunk(200, function ($assignments) use ($handle) {
                foreach ($assignments as $assignment) {
                    fputcsv($handle, [
                        $assignment->ticket?->ticket_number,
                        $assignment->ticket?->subject,
                        $assignment->assignedTo?->name,
                        $assignment->assignedTo?->email,
                        optional($assignment->assigned_at)->format('Y-m-d H:i:s'),
                        $assignment->is_active ? 1 : 0,
                        $assignment->note,
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportTicketAssignmentsRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.ticket-assignments.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return redirect()
                ->route('admin.ticket-assignments.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, [
            'ticket_number',
            'assigned_to',
            'assigned_to_email',
            'assigned_at',
            'is_active',
            'note',
        ]);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $ticketNumber = trim((string) ($data['ticket_number'] ?? ''));
            $assignedEmail = trim((string) ($data['assigned_to_email'] ?? ''));
            $assignedName = trim((string) ($data['assigned_to'] ?? ''));

            if ($ticketNumber === '') {
                $skipped++;
                continue;
            }

            $ticketId = SupportTicket::query()
                ->where('ticket_number', $ticketNumber)
                ->value('id');

            if (! $ticketId) {
                $skipped++;
                continue;
            }

            $assignedTo = null;
            if ($assignedEmail !== '') {
                $assignedTo = User::query()->where('email', $assignedEmail)->value('id');
            }

            if (! $assignedTo && $assignedName !== '') {
                $assignedTo = User::query()->where('name', $assignedName)->value('id');
            }

            if (! $assignedTo) {
                $skipped++;
                continue;
            }

            $assignedAt = $this->parseDateTime($data['assigned_at'] ?? null) ?? now();
            $isActive = $this->parseBool($data['is_active'] ?? null, true);

            DB::transaction(function () use ($ticketId, $assignedTo, $assignedAt, $isActive, $data) {
                if ($isActive) {
                    SupportTicketAssignment::query()
                        ->where('support_ticket_id', $ticketId)
                        ->update(['is_active' => false]);
                }

                SupportTicketAssignment::create([
                    'support_ticket_id' => $ticketId,
                    'assigned_to' => $assignedTo,
                    'note' => $data['note'] ?? null,
                    'is_active' => $isActive,
                    'assigned_at' => $assignedAt,
                ]);
            });

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.ticket-assignments.index')
            ->with('status', "Imported {$imported} assignments. Skipped {$skipped} rows.");
    }

    private function parseDateTime(?string $value): ?Carbon
    {
        if (! $value) {
            return null;
        }

        try {
            return Carbon::parse($value);
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }
}
