<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ImportUsersRequest;
use App\Http\Requests\Admin\StoreUserRequest;
use App\Http\Requests\Admin\UpdateUserRequest;
use App\Models\City;
use App\Models\Country;
use App\Models\Plan;
use App\Models\State;
use App\Models\User;
use App\Support\UserDefaults;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Spatie\Permission\Models\Role;

class UserController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->query('search');
        $roleFilter = $request->query('role');

        $query = User::query()->with('roles')->orderBy('name');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('email', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($roleFilter) {
            $query->whereHas('roles', function ($builder) use ($roleFilter) {
                $builder->where('name', $roleFilter);
            });
        }

        return view('admin.users.index', [
            'users' => $query->paginate(10)->withQueryString(),
            'roles' => Role::query()->orderBy('name')->get(),
            'search' => $search,
            'roleFilter' => $roleFilter,
        ]);
    }

    public function create()
    {
        return view('admin.users.create', [
            'roles' => Role::query()->orderBy('name')->get(),
            'plans' => Plan::query()->orderBy('name')->get(),
            'countries' => Country::query()->where('is_active', true)->orderBy('name')->get(),
            'states' => State::query()->where('is_active', true)->orderBy('name')->get(),
            'cities' => City::query()->where('is_active', true)->orderBy('name')->get(),
        ]);
    }

    public function store(StoreUserRequest $request)
    {
        $data = $request->validated();
        $roles = $data['roles'] ?? [];
        unset($data['roles'], $data['email_verified']);

        $data['password'] = Hash::make($data['password']);
        $data['is_active'] = $request->boolean('is_active');
        $data['email_verified_at'] = $request->boolean('email_verified') ? now() : null;
        if (empty($data['plan_id'])) {
            $data['plan_id'] = UserDefaults::defaultPlanId();
        }

        if ($request->hasFile('profile_photo')) {
            $data['profile_photo_path'] = $request->file('profile_photo')->store('avatars', 'public');
        }

        $user = User::create($data);
        $user->syncRoles($roles);
        if (empty($roles)) {
            UserDefaults::assignDefaultRole($user);
        }

        if ($request->has('save_and_new')) {
            return redirect()
                ->route('admin.users.create')
                ->with('status', 'User created. You can add another.');
        }

        return redirect()
            ->route('admin.users.index')
            ->with('status', 'User created successfully.');
    }

    public function show(User $user)
    {
        $user->load(['roles', 'plan', 'country', 'state', 'city']);

        return view('admin.users.show', [
            'user' => $user,
        ]);
    }

    public function edit(User $user)
    {
        $user->load('roles');

        return view('admin.users.edit', [
            'user' => $user,
            'roles' => Role::query()->orderBy('name')->get(),
            'plans' => Plan::query()->orderBy('name')->get(),
            'countries' => Country::query()->where('is_active', true)->orderBy('name')->get(),
            'states' => State::query()->where('is_active', true)->orderBy('name')->get(),
            'cities' => City::query()->where('is_active', true)->orderBy('name')->get(),
        ]);
    }

    public function update(UpdateUserRequest $request, User $user)
    {
        $data = $request->validated();
        $roles = $data['roles'] ?? [];
        unset($data['roles'], $data['email_verified']);

        $data['is_active'] = $request->boolean('is_active');
        $data['email_verified_at'] = $request->boolean('email_verified') ? now() : null;

        if (! empty($data['password'])) {
            $data['password'] = Hash::make($data['password']);
        } else {
            unset($data['password']);
        }

        if ($request->hasFile('profile_photo')) {
            if ($user->profile_photo_path) {
                Storage::disk('public')->delete($user->profile_photo_path);
            }
            $data['profile_photo_path'] = $request->file('profile_photo')->store('avatars', 'public');
        }

        $user->update($data);
        $user->syncRoles($roles);

        return redirect()
            ->route('admin.users.edit', $user)
            ->with('status', 'User updated successfully.');
    }

    public function destroy(User $user)
    {
        if ($user->id === auth()->id()) {
            return redirect()
                ->route('admin.users.index')
                ->with('error', 'You cannot delete your own account.');
        }

        if ($user->profile_photo_path) {
            Storage::disk('public')->delete($user->profile_photo_path);
        }

        $user->delete();

        return redirect()
            ->route('admin.users.index')
            ->with('status', 'User deleted successfully.');
    }

    public function bulkDestroy(Request $request)
    {
        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:users,id'],
        ]);

        $ids = collect($validated['ids'])->unique()->values();

        if ($ids->contains(auth()->id())) {
            return redirect()
                ->route('admin.users.index')
                ->with('error', 'You cannot delete your own account.');
        }

        User::query()->whereIn('id', $ids)->each(function (User $user) {
            if ($user->profile_photo_path) {
                Storage::disk('public')->delete($user->profile_photo_path);
            }
            $user->delete();
        });

        return redirect()
            ->route('admin.users.index')
            ->with('status', 'Selected users deleted successfully.');
    }

    public function export(Request $request)
    {
        $search = $request->query('search');
        $roleFilter = $request->query('role');

        $query = User::query()->with('roles')->orderBy('name');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('email', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($roleFilter) {
            $query->whereHas('roles', function ($builder) use ($roleFilter) {
                $builder->where('name', $roleFilter);
            });
        }

        $filename = 'users-' . now()->format('Ymd-His') . '.csv';

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, [
                'name',
                'email',
                'phone',
                'gender',
                'roles',
                'is_active',
                'email_verified',
            ]);

            $query->chunk(200, function ($users) use ($handle) {
                foreach ($users as $user) {
                    fputcsv($handle, [
                        $user->name,
                        $user->email,
                        $user->phone,
                        $user->gender,
                        $user->roles->pluck('name')->implode('|'),
                        $user->is_active ? 1 : 0,
                        $user->email_verified_at ? 1 : 0,
                    ]);
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function import(ImportUsersRequest $request)
    {
        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return redirect()
                ->route('admin.users.index')
                ->with('error', 'Unable to read the uploaded file.');
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);

            return redirect()
                ->route('admin.users.index')
                ->with('error', 'CSV file is empty.');
        }

        $map = $this->mapHeaders($header, [
            'name',
            'email',
            'phone',
            'gender',
            'password',
            'roles',
            'is_active',
        ]);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ''));
            $email = trim((string) ($data['email'] ?? ''));
            $password = trim((string) ($data['password'] ?? ''));

            if ($name === '' || $email === '' || $password === '') {
                $skipped++;
                continue;
            }

            if (User::query()->where('email', $email)->exists()) {
                $skipped++;
                continue;
            }

            $user = User::create([
                'name' => $name,
                'email' => $email,
                'phone' => trim((string) ($data['phone'] ?? '')) ?: null,
                'gender' => in_array($data['gender'] ?? '', ['male', 'female', 'other'], true) ? $data['gender'] : null,
                'password' => Hash::make($password),
                'is_active' => (bool) ($data['is_active'] ?? true),
                'email_verified_at' => now(),
                'plan_id' => UserDefaults::defaultPlanId(),
            ]);

            $roleNames = array_filter(array_map('trim', explode('|', (string) ($data['roles'] ?? ''))));
            if (! empty($roleNames)) {
                $validRoles = Role::query()->whereIn('name', $roleNames)->pluck('name')->all();
                $user->syncRoles($validRoles);
            }

            if (empty($roleNames)) {
                UserDefaults::assignDefaultRole($user);
            }

            $imported++;
        }

        fclose($handle);

        return redirect()
            ->route('admin.users.index')
            ->with('status', "Imported {$imported} user(s). Skipped {$skipped}.");
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }
}
