<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\UserDeleteRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class UserDeleteRequestController extends Controller
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', UserDeleteRequest::class);

        $status = (string) $request->query('status', UserDeleteRequest::STATUS_PENDING);
        $query = UserDeleteRequest::query()
            ->with(['user', 'processor'])
            ->orderByDesc('created_at');

        if ($status !== '' && $status !== 'all') {
            $query->where('status', $status);
        }

        return view('admin.user-delete-requests.index', [
            'requests' => $query->paginate(10)->withQueryString(),
            'status' => $status,
        ]);
    }

    public function update(Request $request, UserDeleteRequest $userDeleteRequest)
    {
        $this->authorize('update', $userDeleteRequest);

        $data = $request->validate([
            'action' => ['required', 'in:accept,reject'],
        ]);

        if ($userDeleteRequest->status !== UserDeleteRequest::STATUS_PENDING) {
            return back()->with('error', 'This request has already been processed.');
        }

        $userDeleteRequest->status = $data['action'] === 'accept'
            ? UserDeleteRequest::STATUS_ACCEPTED
            : UserDeleteRequest::STATUS_REJECTED;
        $userDeleteRequest->processed_by = $request->user()?->id;
        $userDeleteRequest->processed_at = now();
        $userDeleteRequest->save();

        if ($data['action'] === 'accept') {
            $user = $userDeleteRequest->user;
            if ($user) {
                if ($user->profile_photo_path) {
                    Storage::disk('public')->delete($user->profile_photo_path);
                }
                $user->delete();
            }

            return back()->with('status', 'Delete request accepted. User account has been removed.');
        }

        return back()->with('status', 'Delete request rejected.');
    }
}
