<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportActivityLogsRequest;
use App\Http\Requests\Admin\StoreActivityLogRequest;
use App\Http\Requests\Admin\UpdateActivityLogRequest;
use App\Models\ActivityLog;
use App\Models\User;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;

class ActivityLogController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', ActivityLog::class);

        $search = $request->query('search');
        $action = $request->query('action');
        $userId = $request->query('user');
        $sort = $request->query('sort', 'latest');

        $query = ActivityLog::query()->with('user');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('description', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('subject_type', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('action', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($action) {
            $query->where('action', $action);
        }

        if ($userId) {
            $query->where('user_id', $userId);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString(), [
            'actions' => config('system.activity_types', []),
            'users' => User::query()->orderBy('name')->limit(200)->get(['id', 'name', 'email']),
        ]);
    }

    public function store(StoreActivityLogRequest $request)
    {
        $this->authorize('create', ActivityLog::class);

        $data = $request->validated();

        $log = ActivityLog::create([
            'user_id' => $data['user_id'] ?? null,
            'action' => $data['action'],
            'subject_type' => $data['subject_type'] ?? null,
            'subject_id' => $data['subject_id'] ?? null,
            'description' => $data['description'] ?? null,
            'ip_address' => $data['ip_address'] ?? null,
            'user_agent' => $data['user_agent'] ?? null,
            'logged_at' => $this->parseDateTime($data['logged_at'] ?? null) ?? now(),
        ]);

        return $this->success($log->load('user'), [], 201);
    }

    public function show(ActivityLog $activityLog)
    {
        $this->authorize('view', $activityLog);

        return $this->success($activityLog->load('user'));
    }

    public function update(UpdateActivityLogRequest $request, ActivityLog $activityLog)
    {
        $this->authorize('update', $activityLog);

        $data = $request->validated();

        $activityLog->update([
            'user_id' => $data['user_id'] ?? null,
            'action' => $data['action'],
            'subject_type' => $data['subject_type'] ?? null,
            'subject_id' => $data['subject_id'] ?? null,
            'description' => $data['description'] ?? null,
            'ip_address' => $data['ip_address'] ?? null,
            'user_agent' => $data['user_agent'] ?? null,
            'logged_at' => $this->parseDateTime($data['logged_at'] ?? null),
        ]);

        return $this->success($activityLog->fresh()->load('user'));
    }

    public function destroy(ActivityLog $activityLog)
    {
        $this->authorize('delete', $activityLog);

        $activityLog->delete();

        return $this->success(['message' => 'Activity log deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', ActivityLog::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:activity_logs,id'],
        ]);

        ActivityLog::query()->whereIn('id', $validated['ids'])->delete();

        return $this->success(['message' => 'Selected activity logs deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', ActivityLog::class);

        $search = $request->query('search');
        $action = $request->query('action');
        $userId = $request->query('user');
        $sort = $request->query('sort', 'latest');

        $query = ActivityLog::query()->with('user');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('description', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('subject_type', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('action', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($action) {
            $query->where('action', $action);
        }

        if ($userId) {
            $query->where('user_id', $userId);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'user_email', 'label' => 'User Email'],
            ['key' => 'action', 'label' => 'Action'],
            ['key' => 'subject_type', 'label' => 'Subject Type'],
            ['key' => 'subject_id', 'label' => 'Subject ID'],
            ['key' => 'description', 'label' => 'Description'],
            ['key' => 'ip_address', 'label' => 'IP Address'],
            ['key' => 'user_agent', 'label' => 'User Agent'],
            ['key' => 'logged_at', 'label' => 'Logged At'],
        ];

        $map = fn (ActivityLog $log) => [
            'user_email' => $log->user?->email,
            'action' => $log->action,
            'subject_type' => $log->subject_type,
            'subject_id' => $log->subject_id,
            'description' => $log->description,
            'ip_address' => $log->ip_address,
            'user_agent' => $log->user_agent,
            'logged_at' => optional($log->logged_at)->format('Y-m-d H:i:s'),
        ];

        $filename = 'activity-logs-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Activity Logs', 'Activity log export');
    }

    public function import(ImportActivityLogsRequest $request)
    {
        $this->authorize('import', ActivityLog::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, [
            'user_email',
            'action',
            'subject_type',
            'subject_id',
            'description',
            'ip_address',
            'user_agent',
            'logged_at',
        ]);

        $actions = array_keys(config('system.activity_types', []));

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $action = $this->normalizeAction($data['action'] ?? null, $actions);

            if ($action === '') {
                $skipped++;
                continue;
            }

            $userId = null;
            $userEmail = trim((string) ($data['user_email'] ?? ''));
            if ($userEmail !== '') {
                $userId = User::query()->where('email', $userEmail)->value('id');
            }

            ActivityLog::create([
                'user_id' => $userId,
                'action' => $action,
                'subject_type' => $data['subject_type'] ?? null,
                'subject_id' => $this->parseInt($data['subject_id'] ?? null),
                'description' => $data['description'] ?? null,
                'ip_address' => $data['ip_address'] ?? null,
                'user_agent' => $data['user_agent'] ?? null,
                'logged_at' => $this->parseDateTime($data['logged_at'] ?? null) ?? now(),
            ]);

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function parseDateTime(?string $value): ?Carbon
    {
        if (! $value) {
            return null;
        }

        try {
            return Carbon::parse($value);
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function parseInt($value): ?int
    {
        if ($value === null || trim((string) $value) === '') {
            return null;
        }

        return (int) $value;
    }

    private function normalizeAction(?string $value, array $actions): string
    {
        $value = Str::lower(trim((string) $value));

        return in_array($value, $actions, true) ? $value : '';
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'oldest' => $query->orderBy('logged_at', 'asc'),
            'action_asc' => $query->orderBy('action'),
            'action_desc' => $query->orderBy('action', 'desc'),
            default => $query->orderBy('logged_at', 'desc'),
        };
    }
}
