<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Api\PasswordUpdateRequest;
use App\Http\Requests\Api\ProfileUpdateRequest;
use App\Http\Requests\Api\RegisterRequest;
use App\Http\Requests\Auth\LoginRequest;
use App\Models\User;
use App\Support\ActivityLogger;
use App\Support\SecurityAuditLogger;
use App\Support\SecuritySettings;
use App\Support\TwoFactorService;
use App\Support\UserDefaults;
use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;

class AuthController extends ApiController
{
    public function me(Request $request)
    {
        $user = $request->user();

        if (! $user) {
            return $this->success(null);
        }

        $user->load(['roles', 'country', 'state', 'city']);

        return $this->success($user);
    }

    public function login(LoginRequest $request)
    {
        $request->authenticate();

        $request->session()->regenerate();

        $user = $request->user();

        if ($user instanceof User && $this->shouldRequireTwoFactor($user)) {
            $requiresSetup = ! $user->two_factor_enabled;

            $request->session()->put('two_factor_user_id', $user->id);
            $request->session()->put('two_factor_remember', $request->boolean('remember'));
            $request->session()->put('two_factor_setup', $requiresSetup);

            SecurityAuditLogger::log('two_factor_required', 'info', $user, $request, [
                'setup_required' => $requiresSetup,
            ]);

            Auth::guard('web')->logout();

            return $this->success([
                'two_factor_required' => true,
                'setup_required' => $requiresSetup,
                'email' => $user->email,
            ]);
        }

        if ($user instanceof User) {
            SecurityAuditLogger::log('login_success', 'success', $user, $request);
            ActivityLogger::logFromRequest($request, 'login', 'User', $user->id, 'Logged in');
        }

        return $this->success([
            'two_factor_required' => false,
            'user' => $user?->load(['roles', 'country', 'state', 'city']),
        ]);
    }

    public function logout(Request $request)
    {
        $user = $request->user();

        Auth::guard('web')->logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        if ($user) {
            ActivityLogger::logFromRequest($request, 'logout', 'User', $user->id, 'Logged out');
        }

        return $this->success(['message' => 'Logged out.']);
    }

    public function register(RegisterRequest $request)
    {
        $data = $request->validated();
        $data['password'] = Hash::make($data['password']);
        $data['email_verified_at'] = null;
        $data['plan_id'] = $data['plan_id'] ?? UserDefaults::defaultPlanId();

        $user = User::create($data);
        UserDefaults::assignDefaultRole($user);

        Auth::login($user);
        $request->session()->regenerate();

        return $this->success($user->load(['roles']));
    }

    public function updateProfile(ProfileUpdateRequest $request)
    {
        $user = $request->user();

        if (! $user) {
            return $this->success(null, [], 401);
        }

        $data = $request->validated();

        if ($request->hasFile('profile_photo')) {
            if ($user->profile_photo_path) {
                Storage::disk('public')->delete($user->profile_photo_path);
            }
            $data['profile_photo_path'] = $request->file('profile_photo')->store('avatars', 'public');
        }

        $user->update($data);

        return $this->success($user->fresh()->load(['roles', 'country', 'state', 'city']));
    }

    public function updatePassword(PasswordUpdateRequest $request)
    {
        $user = $request->user();

        if (! $user) {
            return $this->success(null, [], 401);
        }

        $data = $request->validated();

        if (! Hash::check($data['current_password'], $user->password)) {
            throw ValidationException::withMessages([
                'current_password' => 'Current password is incorrect.',
            ]);
        }

        $user->update([
            'password' => Hash::make($data['password']),
        ]);

        return $this->success(['message' => 'Password updated.']);
    }

    public function forgotPassword(Request $request)
    {
        $request->validate([
            'email' => ['required', 'email'],
        ]);

        $status = Password::sendResetLink($request->only('email'));

        if ($status !== Password::RESET_LINK_SENT) {
            throw ValidationException::withMessages([
                'email' => __($status),
            ]);
        }

        return $this->success(['message' => __($status)]);
    }

    public function resetPassword(Request $request)
    {
        $request->validate([
            'token' => ['required', 'string'],
            'email' => ['required', 'email'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);

        $status = Password::reset(
            $request->only('email', 'password', 'password_confirmation', 'token'),
            function (User $user, string $password) {
                $user->forceFill([
                    'password' => Hash::make($password),
                ])->save();

                event(new PasswordReset($user));
            }
        );

        if ($status !== Password::PASSWORD_RESET) {
            throw ValidationException::withMessages([
                'email' => __($status),
            ]);
        }

        return $this->success(['message' => __($status)]);
    }

    public function twoFactorChallenge(Request $request)
    {
        $request->validate([
            'code' => ['required', 'string'],
        ]);

        $user = $this->resolveTwoFactorUser($request);

        if (! $user || ! $user->two_factor_secret) {
            return $this->success(['message' => 'Two-factor verification expired.'], [], 422);
        }

        if (! TwoFactorService::verifyCode($user->two_factor_secret, $request->input('code'))) {
            SecurityAuditLogger::log('two_factor_failed', 'failed', $user, $request);

            throw ValidationException::withMessages([
                'code' => 'The authentication code is invalid.',
            ]);
        }

        $this->finalizeLogin($request, $user);

        SecurityAuditLogger::log('two_factor_verified', 'success', $user, $request);
        SecurityAuditLogger::log('login_success', 'success', $user, $request);
        ActivityLogger::logFromRequest($request, 'login', 'User', $user->id, 'Logged in with 2FA');

        return $this->success([
            'two_factor_required' => false,
            'user' => $user->load(['roles', 'country', 'state', 'city']),
        ]);
    }

    public function twoFactorSetup(Request $request)
    {
        $user = $this->resolveTwoFactorUser($request);

        if (! $user) {
            return $this->success(['message' => 'Two-factor setup expired.'], [], 422);
        }

        $secret = $request->session()->get('two_factor_setup_secret');
        if (! $secret) {
            $secret = TwoFactorService::generateSecret();
            $request->session()->put('two_factor_setup_secret', $secret);
        }

        $issuer = SecuritySettings::getValue('two_factor_issuer', config('app.name', 'SnapNest – Event Photo Sharing Platform with QR & Face Recognition'));
        $otpAuthUrl = TwoFactorService::buildOtpAuthUrl($user, $secret, $issuer);

        return $this->success([
            'secret' => $secret,
            'otpAuthUrl' => $otpAuthUrl,
            'issuer' => $issuer,
            'email' => $user->email,
        ]);
    }

    public function twoFactorSetupStore(Request $request)
    {
        $request->validate([
            'code' => ['required', 'string'],
        ]);

        $user = $this->resolveTwoFactorUser($request);

        if (! $user) {
            return $this->success(['message' => 'Two-factor setup expired.'], [], 422);
        }

        $secret = $request->session()->get('two_factor_setup_secret');

        if (! $secret) {
            return $this->success(['message' => 'Two-factor setup expired.'], [], 422);
        }

        if (! TwoFactorService::verifyCode($secret, $request->input('code'))) {
            SecurityAuditLogger::log('two_factor_failed', 'failed', $user, $request);

            throw ValidationException::withMessages([
                'code' => 'The authentication code is invalid.',
            ]);
        }

        $user->forceFill([
            'two_factor_enabled' => true,
            'two_factor_secret' => $secret,
            'two_factor_confirmed_at' => now(),
        ])->save();

        $request->session()->forget([
            'two_factor_setup_secret',
            'two_factor_setup',
        ]);

        $this->finalizeLogin($request, $user);

        SecurityAuditLogger::log('two_factor_enabled', 'success', $user, $request);
        SecurityAuditLogger::log('login_success', 'success', $user, $request);
        ActivityLogger::logFromRequest($request, 'login', 'User', $user->id, 'Logged in with 2FA setup');

        return $this->success([
            'two_factor_required' => false,
            'user' => $user->load(['roles', 'country', 'state', 'city']),
        ]);
    }

    private function shouldRequireTwoFactor(User $user): bool
    {
        $settings = SecuritySettings::get();
        $adminOnly = (bool) ($settings['two_factor_admin_only'] ?? false);

        if (! $adminOnly) {
            return $user->two_factor_enabled;
        }

        $adminRoles = config('security.admin_roles', []);

        return $user->hasAnyRole($adminRoles) || $user->two_factor_enabled;
    }

    private function resolveTwoFactorUser(Request $request): ?User
    {
        $userId = $request->session()->get('two_factor_user_id');

        if (! $userId) {
            return null;
        }

        return User::query()->find($userId);
    }

    private function finalizeLogin(Request $request, User $user): void
    {
        $remember = (bool) $request->session()->pull('two_factor_remember', false);

        Auth::login($user, $remember);

        $request->session()->forget([
            'two_factor_user_id',
            'two_factor_setup',
        ]);

        $request->session()->regenerate();
    }
}
