<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportBlockedIpsRequest;
use App\Http\Requests\Admin\StoreBlockedIpRequest;
use App\Http\Requests\Admin\UpdateBlockedIpRequest;
use App\Models\BlockedIp;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class BlockedIpController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', BlockedIp::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = BlockedIp::query()->with('creator');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('ip_address', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('reason', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StoreBlockedIpRequest $request)
    {
        $this->authorize('create', BlockedIp::class);

        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');
        $data['created_by'] = $request->user()?->id;

        $blockedIp = BlockedIp::create($data);

        return $this->success($blockedIp->load('creator'), [], 201);
    }

    public function show(BlockedIp $blockedIp)
    {
        $this->authorize('view', $blockedIp);

        return $this->success($blockedIp->load('creator'));
    }

    public function update(UpdateBlockedIpRequest $request, BlockedIp $blockedIp)
    {
        $this->authorize('update', $blockedIp);

        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        $blockedIp->update($data);

        return $this->success($blockedIp->fresh()->load('creator'));
    }

    public function destroy(BlockedIp $blockedIp)
    {
        $this->authorize('delete', $blockedIp);

        $blockedIp->delete();

        return $this->success(['message' => 'Blocked IP deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', BlockedIp::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:blocked_ips,id'],
        ]);

        BlockedIp::query()->whereIn('id', $validated['ids'])->delete();

        return $this->success(['message' => 'Selected blocked IPs deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', BlockedIp::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = BlockedIp::query()->with('creator');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('ip_address', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('reason', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'ip_address', 'label' => 'IP Address'],
            ['key' => 'reason', 'label' => 'Reason'],
            ['key' => 'is_active', 'label' => 'Status'],
            ['key' => 'created_by', 'label' => 'Created By'],
            ['key' => 'created_at', 'label' => 'Created At'],
        ];

        $map = fn (BlockedIp $blockedIp) => [
            'ip_address' => $blockedIp->ip_address,
            'reason' => $blockedIp->reason,
            'is_active' => $blockedIp->is_active ? 'Active' : 'Inactive',
            'created_by' => $blockedIp->creator?->email,
            'created_at' => optional($blockedIp->created_at)->format('Y-m-d H:i:s'),
        ];

        $filename = 'blocked-ips-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Blocked IP Report', 'Blocked IP export');
    }

    public function import(ImportBlockedIpsRequest $request)
    {
        $this->authorize('create', BlockedIp::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, ['ip_address', 'reason', 'is_active']);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $ipAddress = trim((string) ($data['ip_address'] ?? ''));

            if ($ipAddress === '') {
                $skipped++;
                continue;
            }

            BlockedIp::updateOrCreate(
                ['ip_address' => $ipAddress],
                [
                    'reason' => $data['reason'] ?? null,
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'message' => "Imported {$imported} blocked IPs. Skipped {$skipped} rows.",
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'ip_asc' => $query->orderBy('ip_address'),
            'ip_desc' => $query->orderBy('ip_address', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }
}
