<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportBlockedLocationsRequest;
use App\Http\Requests\Admin\StoreBlockedLocationRequest;
use App\Http\Requests\Admin\UpdateBlockedLocationRequest;
use App\Models\BlockedLocation;
use App\Models\City;
use App\Models\Country;
use App\Models\State;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class BlockedLocationController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', BlockedLocation::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $countryId = $request->query('country_id');
        $stateId = $request->query('state_id');
        $cityId = $request->query('city_id');
        $sort = $request->query('sort', 'latest');

        $query = BlockedLocation::query()->with(['country', 'state', 'city', 'creator']);

        if ($search) {
            $query->where('reason', 'like', "%" . $this->escapeLike($search) . "%");
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        if ($countryId) {
            $query->where('country_id', $countryId);
        }

        if ($stateId) {
            $query->where('state_id', $stateId);
        }

        if ($cityId) {
            $query->where('city_id', $cityId);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StoreBlockedLocationRequest $request)
    {
        $this->authorize('create', BlockedLocation::class);

        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');
        $data['created_by'] = $request->user()?->id;

        $blockedLocation = BlockedLocation::create($data);

        return $this->success($blockedLocation->load(['country', 'state', 'city', 'creator']), [], 201);
    }

    public function show(BlockedLocation $blockedLocation)
    {
        $this->authorize('view', $blockedLocation);

        return $this->success($blockedLocation->load(['country', 'state', 'city', 'creator']));
    }

    public function update(UpdateBlockedLocationRequest $request, BlockedLocation $blockedLocation)
    {
        $this->authorize('update', $blockedLocation);

        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        $blockedLocation->update($data);

        return $this->success($blockedLocation->fresh()->load(['country', 'state', 'city', 'creator']));
    }

    public function destroy(BlockedLocation $blockedLocation)
    {
        $this->authorize('delete', $blockedLocation);

        $blockedLocation->delete();

        return $this->success(['message' => 'Blocked location deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', BlockedLocation::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:blocked_locations,id'],
        ]);

        BlockedLocation::query()->whereIn('id', $validated['ids'])->delete();

        return $this->success(['message' => 'Selected blocked locations deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', BlockedLocation::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $countryId = $request->query('country_id');
        $stateId = $request->query('state_id');
        $cityId = $request->query('city_id');
        $sort = $request->query('sort', 'latest');

        $query = BlockedLocation::query()->with(['country', 'state', 'city', 'creator']);

        if ($search) {
            $query->where('reason', 'like', "%" . $this->escapeLike($search) . "%");
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        if ($countryId) {
            $query->where('country_id', $countryId);
        }

        if ($stateId) {
            $query->where('state_id', $stateId);
        }

        if ($cityId) {
            $query->where('city_id', $cityId);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'country', 'label' => 'Country'],
            ['key' => 'state', 'label' => 'State'],
            ['key' => 'city', 'label' => 'City'],
            ['key' => 'reason', 'label' => 'Reason'],
            ['key' => 'is_active', 'label' => 'Status'],
            ['key' => 'created_by', 'label' => 'Created By'],
            ['key' => 'created_at', 'label' => 'Created At'],
        ];

        $map = fn (BlockedLocation $blockedLocation) => [
            'country' => $blockedLocation->country?->name,
            'state' => $blockedLocation->state?->name,
            'city' => $blockedLocation->city?->name,
            'reason' => $blockedLocation->reason,
            'is_active' => $blockedLocation->is_active ? 'Active' : 'Inactive',
            'created_by' => $blockedLocation->creator?->email,
            'created_at' => optional($blockedLocation->created_at)->format('Y-m-d H:i:s'),
        ];

        $filename = 'blocked-locations-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Blocked Locations Report', 'Blocked locations export');
    }

    public function import(ImportBlockedLocationsRequest $request)
    {
        $this->authorize('create', BlockedLocation::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, ['country', 'state', 'city', 'reason', 'is_active']);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $countryName = trim((string) ($data['country'] ?? ''));
            $stateName = trim((string) ($data['state'] ?? ''));
            $cityName = trim((string) ($data['city'] ?? ''));

            $countryId = $countryName !== ''
                ? Country::query()->where('name', $countryName)->value('id')
                : null;
            $stateId = $stateName !== ''
                ? State::query()->where('name', $stateName)->value('id')
                : null;
            $cityId = $cityName !== ''
                ? City::query()->where('name', $cityName)->value('id')
                : null;

            if (! $countryId && ! $stateId && ! $cityId) {
                $skipped++;
                continue;
            }

            BlockedLocation::updateOrCreate(
                [
                    'country_id' => $countryId,
                    'state_id' => $stateId,
                    'city_id' => $cityId,
                ],
                [
                    'reason' => $data['reason'] ?? null,
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'message' => "Imported {$imported} blocked locations. Skipped {$skipped} rows.",
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'location_asc' => $query->orderBy('country_id')->orderBy('state_id')->orderBy('city_id'),
            'location_desc' => $query->orderBy('country_id', 'desc')->orderBy('state_id', 'desc')->orderBy('city_id', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }
}
