<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportBlogCategoriesRequest;
use App\Http\Requests\Admin\StoreBlogCategoryRequest;
use App\Http\Requests\Admin\UpdateBlogCategoryRequest;
use App\Models\BlogCategory;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class BlogCategoryController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', BlogCategory::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = BlogCategory::query()->withCount('posts');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StoreBlogCategoryRequest $request)
    {
        $this->authorize('create', BlogCategory::class);

        $data = $request->validated();

        $category = BlogCategory::create([
            'name' => $data['name'],
            'slug' => $this->uniqueSlug($data['name']),
            'is_active' => $request->boolean('is_active'),
            'created_by' => $request->user()?->id,
        ]);

        return $this->success($category, [], 201);
    }

    public function show(BlogCategory $blogCategory)
    {
        $this->authorize('view', $blogCategory);

        $blogCategory->loadCount('posts');

        return $this->success($blogCategory);
    }

    public function update(UpdateBlogCategoryRequest $request, BlogCategory $blogCategory)
    {
        $this->authorize('update', $blogCategory);

        $data = $request->validated();

        $payload = [
            'name' => $data['name'],
            'is_active' => $request->boolean('is_active'),
        ];

        if ($data['name'] !== $blogCategory->name) {
            $payload['slug'] = $this->uniqueSlug($data['name'], $blogCategory->id);
        }

        $blogCategory->update($payload);

        return $this->success($blogCategory->fresh()->loadCount('posts'));
    }

    public function destroy(BlogCategory $blogCategory)
    {
        $this->authorize('delete', $blogCategory);

        $blogCategory->delete();

        return $this->success(['message' => 'Category deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', BlogCategory::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:blog_categories,id'],
        ]);

        BlogCategory::query()->whereIn('id', $validated['ids'])->delete();

        return $this->success(['message' => 'Selected categories deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', BlogCategory::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = BlogCategory::query()->withCount('posts');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'name', 'label' => 'Name'],
            ['key' => 'slug', 'label' => 'Slug'],
            ['key' => 'posts_count', 'label' => 'Posts Count'],
            ['key' => 'is_active', 'label' => 'Status'],
        ];

        $map = fn (BlogCategory $category) => [
            'name' => $category->name,
            'slug' => $category->slug,
            'posts_count' => $category->posts_count,
            'is_active' => $category->is_active ? 'Active' : 'Inactive',
        ];

        $filename = 'blog-categories-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Blog Categories Report', 'Blog category export');
    }

    public function import(ImportBlogCategoriesRequest $request)
    {
        $this->authorize('create', BlogCategory::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, ['name', 'is_active']);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ''));

            if ($name === '') {
                $skipped++;
                continue;
            }

            $category = BlogCategory::query()->where('name', $name)->first();

            if ($category) {
                $category->update([
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]);
            } else {
                BlogCategory::create([
                    'name' => $name,
                    'slug' => $this->uniqueSlug($name),
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]);
            }

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'message' => "Imported {$imported} categories. Skipped {$skipped} rows.",
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_desc' => $query->orderBy('name', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('name'),
        };
    }

    private function uniqueSlug(string $name, ?int $ignoreId = null): string
    {
        $base = Str::slug($name);
        $slug = $base !== '' ? $base : Str::random(8);
        $counter = 1;

        while ($this->slugExists($slug, $ignoreId)) {
            $slug = $base . '-' . $counter;
            $counter++;
        }

        return $slug;
    }

    private function slugExists(string $slug, ?int $ignoreId): bool
    {
        $query = BlogCategory::query()->where('slug', $slug);
        if ($ignoreId) {
            $query->where('id', '!=', $ignoreId);
        }

        return $query->exists();
    }
}
