<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportBlogPostsRequest;
use App\Http\Requests\Admin\StoreBlogPostRequest;
use App\Http\Requests\Admin\UpdateBlogPostRequest;
use App\Models\BlogCategory;
use App\Models\BlogPost;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class BlogPostController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', BlogPost::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $categoryId = $request->query('category_id');
        $bookmark = $request->query('bookmark');
        $sort = $request->query('sort', 'published_desc');

        $query = BlogPost::query()->with(['category', 'creator']);

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('title', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('author_name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhereHas('category', function ($categoryQuery) use ($search) {
                        $categoryQuery->where('name', 'like', "%" . $this->escapeLike($search) . "%");
                    });
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        if ($categoryId) {
            $query->where('category_id', $categoryId);
        }

        if ($bookmark === 'bookmarked') {
            $query->where('is_bookmarked', true);
        } elseif ($bookmark === 'unbookmarked') {
            $query->where('is_bookmarked', false);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StoreBlogPostRequest $request)
    {
        $this->authorize('create', BlogPost::class);

        $data = $request->validated();

        $payload = [
            'title' => $data['title'],
            'slug' => $this->uniqueSlug($data['title']),
            'author_name' => $data['author_name'] ?? null,
            'category_id' => $data['category_id'],
            'excerpt' => $data['excerpt'] ?? null,
            'details' => $data['details'] ?? null,
            'tags' => $this->normalizeTags($data['tags'] ?? null),
            'is_bookmarked' => $request->boolean('is_bookmarked'),
            'is_active' => $request->boolean('is_active'),
            'published_at' => $this->parseDateTime($data['published_at'] ?? null) ?? now(),
            'created_by' => $request->user()?->id,
        ];

        if ($request->hasFile('image')) {
            $payload['image'] = $request->file('image')->store('content/blog', 'public');
        }

        $post = BlogPost::create($payload);

        return $this->success($post->load(['category', 'creator']), [], 201);
    }

    public function show(BlogPost $blogPost)
    {
        $this->authorize('view', $blogPost);

        return $this->success($blogPost->load(['category', 'creator']));
    }

    public function update(UpdateBlogPostRequest $request, BlogPost $blogPost)
    {
        $this->authorize('update', $blogPost);

        $data = $request->validated();

        $payload = [
            'title' => $data['title'],
            'author_name' => $data['author_name'] ?? null,
            'category_id' => $data['category_id'],
            'excerpt' => $data['excerpt'] ?? null,
            'details' => $data['details'] ?? null,
            'tags' => $this->normalizeTags($data['tags'] ?? null),
            'is_bookmarked' => $request->boolean('is_bookmarked'),
            'is_active' => $request->boolean('is_active'),
            'published_at' => $this->parseDateTime($data['published_at'] ?? null),
        ];

        if ($data['title'] !== $blogPost->title) {
            $payload['slug'] = $this->uniqueSlug($data['title'], $blogPost->id);
        }

        if ($request->hasFile('image')) {
            if ($blogPost->image) {
                Storage::disk('public')->delete($blogPost->image);
            }
            $payload['image'] = $request->file('image')->store('content/blog', 'public');
        }

        $blogPost->update($payload);

        return $this->success($blogPost->fresh()->load(['category', 'creator']));
    }

    public function destroy(BlogPost $blogPost)
    {
        $this->authorize('delete', $blogPost);

        if ($blogPost->image) {
            Storage::disk('public')->delete($blogPost->image);
        }

        $blogPost->delete();

        return $this->success(['message' => 'Blog post deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', BlogPost::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:blog_posts,id'],
        ]);

        $posts = BlogPost::query()->whereIn('id', $validated['ids'])->get();

        foreach ($posts as $post) {
            if ($post->image) {
                Storage::disk('public')->delete($post->image);
            }
            $post->delete();
        }

        return $this->success(['message' => 'Selected blog posts deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', BlogPost::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $categoryId = $request->query('category_id');
        $bookmark = $request->query('bookmark');
        $sort = $request->query('sort', 'published_desc');

        $query = BlogPost::query()->with('category');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('title', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('author_name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhereHas('category', function ($categoryQuery) use ($search) {
                        $categoryQuery->where('name', 'like', "%" . $this->escapeLike($search) . "%");
                    });
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        if ($categoryId) {
            $query->where('category_id', $categoryId);
        }

        if ($bookmark === 'bookmarked') {
            $query->where('is_bookmarked', true);
        } elseif ($bookmark === 'unbookmarked') {
            $query->where('is_bookmarked', false);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'title', 'label' => 'Title'],
            ['key' => 'slug', 'label' => 'Slug'],
            ['key' => 'category', 'label' => 'Category'],
            ['key' => 'author_name', 'label' => 'Author'],
            ['key' => 'tags', 'label' => 'Tags'],
            ['key' => 'is_bookmarked', 'label' => 'Bookmarked'],
            ['key' => 'is_active', 'label' => 'Status'],
            ['key' => 'published_at', 'label' => 'Published'],
        ];

        $map = fn (BlogPost $post) => [
            'title' => $post->title,
            'slug' => $post->slug,
            'category' => $post->category?->name,
            'author_name' => $post->author_name,
            'tags' => $post->tags ? implode('|', $post->tags) : '',
            'is_bookmarked' => $post->is_bookmarked ? 'Yes' : 'No',
            'is_active' => $post->is_active ? 'Active' : 'Inactive',
            'published_at' => optional($post->published_at)->format('Y-m-d H:i:s'),
        ];

        $filename = 'blog-posts-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Blog Posts Report', 'Blog post export');
    }

    public function import(ImportBlogPostsRequest $request)
    {
        $this->authorize('create', BlogPost::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, [
            'title',
            'author_name',
            'category',
            'excerpt',
            'details',
            'tags',
            'is_bookmarked',
            'is_active',
            'published_at',
        ]);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $title = trim((string) ($data['title'] ?? ''));
            $categoryName = trim((string) ($data['category'] ?? ''));

            if ($title === '' || $categoryName === '') {
                $skipped++;
                continue;
            }

            $categoryId = BlogCategory::query()->where('name', $categoryName)->value('id');

            if (! $categoryId) {
                $skipped++;
                continue;
            }

            $payload = [
                'author_name' => $data['author_name'] ?? null,
                'category_id' => $categoryId,
                'excerpt' => $data['excerpt'] ?? null,
                'details' => $data['details'] ?? null,
                'tags' => $this->normalizeTags($data['tags'] ?? null),
                'is_bookmarked' => $this->parseBool($data['is_bookmarked'] ?? null, false),
                'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                'published_at' => $this->parseDateTime($data['published_at'] ?? null) ?? now(),
            ];

            $post = BlogPost::query()
                ->where('title', $title)
                ->where('category_id', $categoryId)
                ->first();

            if ($post) {
                $post->update($payload);
            } else {
                BlogPost::create(array_merge($payload, [
                    'title' => $title,
                    'slug' => $this->uniqueSlug($title),
                ]));
            }

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'message' => "Imported {$imported} blog posts. Skipped {$skipped} rows.",
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function normalizeTags(?string $value): ?array
    {
        if (! $value) {
            return null;
        }

        $tags = collect(preg_split('/[|,]/', $value))
            ->map(fn ($tag) => trim($tag))
            ->filter()
            ->unique()
            ->values()
            ->all();

        return $tags ?: null;
    }

    private function parseDateTime(?string $value): ?Carbon
    {
        if (! $value) {
            return null;
        }

        try {
            return Carbon::parse($value);
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'published_asc' => $query->orderBy('published_at'),
            'title_asc' => $query->orderBy('title'),
            'title_desc' => $query->orderBy('title', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('published_at', 'desc'),
        };
    }

    private function uniqueSlug(string $title, ?int $ignoreId = null): string
    {
        $base = Str::slug($title);
        $slug = $base !== '' ? $base : Str::random(8);
        $counter = 1;

        while ($this->slugExists($slug, $ignoreId)) {
            $slug = $base . '-' . $counter;
            $counter++;
        }

        return $slug;
    }

    private function slugExists(string $slug, ?int $ignoreId): bool
    {
        $query = BlogPost::query()->where('slug', $slug);
        if ($ignoreId) {
            $query->where('id', '!=', $ignoreId);
        }

        return $query->exists();
    }
}
