<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportCitiesRequest;
use App\Http\Requests\Admin\StoreCityRequest;
use App\Http\Requests\Admin\UpdateCityRequest;
use App\Models\City;
use App\Models\State;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class CityController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', City::class);

        $search = $request->query('search');
        $stateId = $request->query('state_id');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = City::query()->with('state.country');

        if ($search) {
            $query->where('name', 'like', "%" . $this->escapeLike($search) . "%");
        }

        if ($stateId) {
            $query->where('state_id', $stateId);
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StoreCityRequest $request)
    {
        $this->authorize('create', City::class);

        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        $city = City::create($data);

        return $this->success($city->load('state.country'), [], 201);
    }

    public function show(City $city)
    {
        $this->authorize('view', $city);

        return $this->success($city->load('state.country'));
    }

    public function update(UpdateCityRequest $request, City $city)
    {
        $this->authorize('update', $city);

        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        $city->update($data);

        return $this->success($city->fresh()->load('state.country'));
    }

    public function destroy(City $city)
    {
        $this->authorize('delete', $city);

        $city->delete();

        return $this->success(['message' => 'City deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', City::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:cities,id'],
        ]);

        City::query()->whereIn('id', $validated['ids'])->delete();

        return $this->success(['message' => 'Selected cities deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', City::class);

        $search = $request->query('search');
        $stateId = $request->query('state_id');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = City::query()->with('state.country');

        if ($search) {
            $query->where('name', 'like', "%" . $this->escapeLike($search) . "%");
        }

        if ($stateId) {
            $query->where('state_id', $stateId);
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'name', 'label' => 'Name'],
            ['key' => 'state', 'label' => 'State'],
            ['key' => 'country', 'label' => 'Country'],
            ['key' => 'is_active', 'label' => 'Active'],
        ];

        $map = fn (City $city) => [
            'name' => $city->name,
            'state' => $city->state?->name ?? '-',
            'country' => $city->state?->country?->name ?? '-',
            'is_active' => $city->is_active ? 'Yes' : 'No',
        ];

        $filename = 'cities-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Cities', 'City export');
    }

    public function import(ImportCitiesRequest $request)
    {
        $this->authorize('import', City::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, ['state_id', 'state', 'state_code', 'city', 'name', 'is_active']);
        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ($data['city'] ?? '')));

            $stateId = $this->resolveStateId($data);

            if ($name === '' || ! $stateId) {
                $skipped++;
                continue;
            }

            City::updateOrCreate(
                ['state_id' => $stateId, 'name' => $name],
                ['is_active' => $this->parseBool($data['is_active'] ?? null, true)]
            );

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_desc' => $query->orderBy('name', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('name'),
        };
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function resolveStateId(array $data): ?int
    {
        if (! empty($data['state_id'])) {
            return (int) $data['state_id'];
        }

        $stateName = trim((string) ($data['state'] ?? ''));
        $stateCode = trim((string) ($data['state_code'] ?? ''));

        $query = State::query();

        if ($stateCode !== '') {
            $state = $query->where('code', $stateCode)->first();
            if ($state) {
                return $state->id;
            }
        }

        if ($stateName !== '') {
            return (int) State::query()->where('name', $stateName)->value('id');
        }

        return null;
    }
}
