<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportCouponsRequest;
use App\Http\Requests\Admin\StoreCouponRequest;
use App\Http\Requests\Admin\UpdateCouponRequest;
use App\Models\Coupon;
use App\Models\Plan;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class CouponController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', Coupon::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $type = $request->query('type');
        $sort = $request->query('sort', 'latest');

        $query = Coupon::query()->with('plans');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('code', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($type) {
            $query->where('discount_type', $type);
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StoreCouponRequest $request)
    {
        $this->authorize('create', Coupon::class);

        $data = $request->validated();
        $planIds = $data['plans'] ?? [];
        unset($data['plans']);

        $data['is_active'] = $request->boolean('is_active');
        $data['new_users_only'] = $request->boolean('new_users_only');
        $data['created_by'] = $request->user()?->id;

        $coupon = Coupon::create($data);

        if ($planIds) {
            $coupon->plans()->sync($planIds);
        }

        return $this->success($coupon->load('plans'), [], 201);
    }

    public function show(Coupon $coupon)
    {
        $this->authorize('view', $coupon);

        return $this->success($coupon->load('plans'));
    }

    public function update(UpdateCouponRequest $request, Coupon $coupon)
    {
        $this->authorize('update', $coupon);

        $data = $request->validated();
        $planIds = $data['plans'] ?? [];
        unset($data['plans']);

        $data['is_active'] = $request->boolean('is_active');
        $data['new_users_only'] = $request->boolean('new_users_only');

        $coupon->update($data);
        $coupon->plans()->sync($planIds);

        return $this->success($coupon->fresh()->load('plans'));
    }

    public function destroy(Coupon $coupon)
    {
        $this->authorize('delete', $coupon);

        $coupon->plans()->detach();
        $coupon->delete();

        return $this->success(['message' => 'Coupon deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', Coupon::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:coupons,id'],
        ]);

        $coupons = Coupon::query()->whereIn('id', $validated['ids'])->get();

        foreach ($coupons as $coupon) {
            $coupon->plans()->detach();
            $coupon->delete();
        }

        return $this->success(['message' => 'Selected coupons deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', Coupon::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $type = $request->query('type');
        $sort = $request->query('sort', 'latest');

        $query = Coupon::query()->orderBy('created_at', 'desc');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('code', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($type) {
            $query->where('discount_type', $type);
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'name', 'label' => 'Name'],
            ['key' => 'code', 'label' => 'Code'],
            ['key' => 'discount_type', 'label' => 'Type'],
            ['key' => 'discount_value', 'label' => 'Discount'],
            ['key' => 'min_amount', 'label' => 'Min Amount'],
            ['key' => 'starts_at', 'label' => 'Starts'],
            ['key' => 'ends_at', 'label' => 'Ends'],
            ['key' => 'new_users_only', 'label' => 'New Users Only'],
            ['key' => 'status', 'label' => 'Status'],
        ];

        $map = fn (Coupon $coupon) => [
            'name' => $coupon->name,
            'code' => $coupon->code,
            'discount_type' => $coupon->discount_type,
            'discount_value' => (string) $coupon->discount_value,
            'min_amount' => (string) $coupon->min_amount,
            'starts_at' => optional($coupon->starts_at)->format('Y-m-d'),
            'ends_at' => optional($coupon->ends_at)->format('Y-m-d'),
            'new_users_only' => $coupon->new_users_only ? 'Yes' : 'No',
            'status' => $coupon->is_active ? 'Active' : 'Inactive',
        ];

        $filename = 'coupons-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Coupons', 'Coupon export');
    }

    public function import(ImportCouponsRequest $request)
    {
        $this->authorize('import', Coupon::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, [
            'name',
            'code',
            'discount_type',
            'discount_value',
            'min_amount',
            'starts_at',
            'ends_at',
            'new_users_only',
            'is_active',
            'plans',
        ]);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ''));
            $code = trim((string) ($data['code'] ?? ''));

            if ($name === '' || $code === '') {
                $skipped++;
                continue;
            }

            $coupon = Coupon::updateOrCreate(
                ['code' => strtoupper($code)],
                [
                    'name' => $name,
                    'discount_type' => $data['discount_type'] ?: 'percent',
                    'discount_value' => (float) ($data['discount_value'] ?? 0),
                    'min_amount' => (float) ($data['min_amount'] ?? 0),
                    'starts_at' => $data['starts_at'] ?: null,
                    'ends_at' => $data['ends_at'] ?: null,
                    'new_users_only' => $this->parseBool($data['new_users_only'] ?? null, false),
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $plans = $this->resolvePlans($data['plans'] ?? null);
            if ($plans) {
                $coupon->plans()->sync($plans);
            }

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_asc' => $query->orderBy('name'),
            'name_desc' => $query->orderBy('name', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function resolvePlans(?string $value): array
    {
        if (! $value) {
            return [];
        }

        $names = collect(preg_split('/[|,]/', $value))
            ->map(fn ($item) => trim($item))
            ->filter()
            ->values()
            ->all();

        if (! $names) {
            return [];
        }

        return Plan::query()
            ->whereIn('name', $names)
            ->pluck('id')
            ->all();
    }
}
