<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportCurrenciesRequest;
use App\Http\Requests\Admin\StoreCurrencyRequest;
use App\Http\Requests\Admin\UpdateCurrencyRequest;
use App\Models\Currency;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class CurrencyController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', Currency::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = Currency::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('code', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('symbol', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StoreCurrencyRequest $request)
    {
        $this->authorize('create', Currency::class);

        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        $currency = Currency::create($data);

        return $this->success($currency, [], 201);
    }

    public function show(Currency $currency)
    {
        $this->authorize('view', $currency);

        return $this->success($currency);
    }

    public function update(UpdateCurrencyRequest $request, Currency $currency)
    {
        $this->authorize('update', $currency);

        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        $currency->update($data);

        return $this->success($currency->fresh());
    }

    public function destroy(Currency $currency)
    {
        $this->authorize('delete', $currency);

        $currency->delete();

        return $this->success(['message' => 'Currency deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', Currency::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:currencies,id'],
        ]);

        Currency::query()->whereIn('id', $validated['ids'])->delete();

        return $this->success(['message' => 'Selected currencies deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', Currency::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = Currency::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('code', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('symbol', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'name', 'label' => 'Name'],
            ['key' => 'code', 'label' => 'Code'],
            ['key' => 'symbol', 'label' => 'Symbol'],
            ['key' => 'precision', 'label' => 'Precision'],
            ['key' => 'is_active', 'label' => 'Active'],
        ];

        $map = fn (Currency $currency) => [
            'name' => $currency->name,
            'code' => $currency->code,
            'symbol' => $currency->symbol,
            'precision' => $currency->precision,
            'is_active' => $currency->is_active ? 'Yes' : 'No',
        ];

        $filename = 'currencies-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Currencies', 'Currency export');
    }

    public function import(ImportCurrenciesRequest $request)
    {
        $this->authorize('import', Currency::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, ['name', 'code', 'symbol', 'precision', 'is_active']);
        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ''));
            $code = strtoupper(trim((string) ($data['code'] ?? '')));

            if ($name === '' || $code === '') {
                $skipped++;
                continue;
            }

            Currency::updateOrCreate(
                ['code' => $code],
                [
                    'name' => $name,
                    'symbol' => trim((string) ($data['symbol'] ?? '')) ?: null,
                    'precision' => (int) ($data['precision'] ?? 2),
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_desc' => $query->orderBy('name', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('name'),
        };
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }
}
