<?php

namespace App\Http\Controllers\Api;

use App\Models\Event;
use App\Models\EventMedia;
use App\Models\Plan;
use App\Models\SupportTicket;
use App\Models\User;
use App\Services\StorageUsageService;
use Illuminate\Http\Request;

class DashboardController extends ApiController
{
    public function index(Request $request)
    {
        $start = $request->query('start_date');
        $end = $request->query('end_date');

        $startDate = $start ? now()->parse($start)->startOfDay() : now()->subDays(30)->startOfDay();
        $endDate = $end ? now()->parse($end)->endOfDay() : now()->endOfDay();

        $eventsQuery = Event::query()->whereBetween('created_at', [$startDate, $endDate]);
        $usersQuery = User::query()->whereBetween('created_at', [$startDate, $endDate]);
        $mediaQuery = EventMedia::query()->whereBetween('created_at', [$startDate, $endDate]);

        $storageSummary = app(StorageUsageService::class)->summary();
        $storageUsedBytes = $storageSummary['used_bytes'];
        $storageLimitBytes = $storageSummary['limit_bytes'];

        $metrics = [
            'events' => $eventsQuery->count(),
            'users' => $usersQuery->count(),
            'uploads' => $mediaQuery->count(),
            'tickets' => SupportTicket::query()->whereBetween('created_at', [$startDate, $endDate])->count(),
            'plans' => Plan::query()->count(),
            'storage_used' => $storageUsedBytes,
            'storage_limit' => $storageLimitBytes,
        ];

        $trend = $this->trend($startDate, $endDate);

        return $this->success([
            'metrics' => $metrics,
            'trend' => $trend,
        ]);
    }

    private function trend($startDate, $endDate): array
    {
        $days = $startDate->diffInDays($endDate) + 1;
        $days = max($days, 1);

        $uploads = EventMedia::query()
            ->whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('DATE(created_at) as day, COUNT(*) as total')
            ->groupBy('day')
            ->pluck('total', 'day')
            ->all();

        $events = Event::query()
            ->whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('DATE(created_at) as day, COUNT(*) as total')
            ->groupBy('day')
            ->pluck('total', 'day')
            ->all();

        $users = User::query()
            ->whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('DATE(created_at) as day, COUNT(*) as total')
            ->groupBy('day')
            ->pluck('total', 'day')
            ->all();

        $labels = [];
        $uploadSeries = [];
        $eventSeries = [];
        $userSeries = [];

        for ($i = 0; $i < $days; $i++) {
            $day = $startDate->copy()->addDays($i)->toDateString();
            $labels[] = $day;
            $uploadSeries[] = (int) ($uploads[$day] ?? 0);
            $eventSeries[] = (int) ($events[$day] ?? 0);
            $userSeries[] = (int) ($users[$day] ?? 0);
        }

        return [
            'labels' => $labels,
            'uploads' => $uploadSeries,
            'events' => $eventSeries,
            'users' => $userSeries,
        ];
    }
}
