<?php

namespace App\Http\Controllers\Api;

use App\Exceptions\PlanLimitReachedException;
use App\Exceptions\StorageLimitReachedException;
use App\Http\Requests\Admin\ChunkEventMediaRequest;
use App\Http\Requests\Admin\StoreEventMediaRequest;
use App\Models\Event;
use App\Models\EventMedia;
use App\Services\EventMediaService;
use App\Services\StorageUsageService;
use App\Support\EventMediaSettings;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;

class EventMediaController extends ApiController
{
    public function index(Request $request, Event $event)
    {
        $this->authorizeEventAccess($request, $event, 'view');

        $sort = $request->query('sort', 'newest');
        $type = $request->query('type');
        $status = $request->query('status');

        $query = $event->media()->orderBy('created_at', 'desc');

        if ($type) {
            $query->where('file_type', $type);
        }

        if ($status) {
            $query->where('status', $status);
        } elseif (! $request->user()) {
            $query->whereIn('status', ['ready', 'processing', 'pending']);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(24)->withQueryString());
    }

    public function store(StoreEventMediaRequest $request, Event $event, EventMediaService $service)
    {
        $this->authorizeEventAccess($request, $event, 'update');

        $files = $request->file('media', []);
        $uploaded = [];

        try {
            foreach ($files as $file) {
                $uploaded[] = $service->storeUploadedFile($event, $file);
            }
        } catch (PlanLimitReachedException $exception) {
            return $this->error($this->planLimitMessage($exception), 422);
        } catch (StorageLimitReachedException $exception) {
            return $this->error($this->storageLimitMessage($exception), 422);
        }

        return $this->success([
            'uploaded' => count($uploaded),
            'media' => $uploaded,
        ], [], 201);
    }

    public function storeChunk(ChunkEventMediaRequest $request, Event $event, EventMediaService $service)
    {
        $this->authorizeEventAccess($request, $event, 'update');

        $uploadId = preg_replace('/[^a-zA-Z0-9_-]/', '', $request->input('upload_id'));
        $chunkIndex = (int) $request->input('chunk_index');
        $totalChunks = (int) $request->input('total_chunks');
        $fileName = $request->input('file_name');
        $chunkFile = $request->file('chunk');

        // Limit total chunks to prevent abuse
        if ($totalChunks > 1000) {
            return $this->error('Too many chunks. Maximum is 1000.', 422);
        }

        // Verify chunk checksum if provided
        $expectedChecksum = $request->input('chunk_checksum');
        if ($expectedChecksum) {
            $actualChecksum = hash_file('sha256', $chunkFile->getRealPath());
            if ($actualChecksum !== $expectedChecksum) {
                return $this->error('Chunk integrity check failed. Please retry upload.', 422);
            }
        }

        $chunkDirectory = 'tmp/event-chunks/' . $uploadId;
        Storage::disk('local')->makeDirectory($chunkDirectory);
        Storage::disk('local')->putFileAs($chunkDirectory, $chunkFile, 'chunk_' . $chunkIndex);

        if ($chunkIndex + 1 < $totalChunks) {
            return $this->success([
                'status' => 'chunk_received',
                'received' => $chunkIndex + 1,
                'total' => $totalChunks,
            ]);
        }

        $assembledPath = storage_path('app/' . $chunkDirectory . '/assembled_' . uniqid());
        $output = fopen($assembledPath, 'wb');

        for ($i = 0; $i < $totalChunks; $i++) {
            $chunkPath = Storage::disk('local')->path($chunkDirectory . '/chunk_' . $i);
            $input = fopen($chunkPath, 'rb');
            stream_copy_to_stream($input, $output);
            fclose($input);
        }

        fclose($output);

        try {
            $media = $service->storeLocalFile($event, $assembledPath, $fileName);
        } catch (PlanLimitReachedException $exception) {
            Storage::disk('local')->deleteDirectory($chunkDirectory);
            if (is_file($assembledPath)) {
                unlink($assembledPath);
            }

            return $this->error($this->planLimitMessage($exception), 422);
        } catch (StorageLimitReachedException $exception) {
            Storage::disk('local')->deleteDirectory($chunkDirectory);
            if (is_file($assembledPath)) {
                unlink($assembledPath);
            }

            return $this->error($this->storageLimitMessage($exception), 422);
        }

        Storage::disk('local')->deleteDirectory($chunkDirectory);
        if (is_file($assembledPath)) {
            unlink($assembledPath);
        }

        return $this->success([
            'status' => 'completed',
            'media_id' => $media?->id,
        ]);
    }

    public function destroy(Request $request, Event $event, EventMedia $media)
    {
        $this->authorizeEventAccess($request, $event, 'update');

        if ($media->event_id !== $event->id) {
            return $this->error('Media not found.', 404);
        }

        $media->delete();

        return $this->success(['message' => 'Media deleted.']);
    }

    public function bulkDestroy(Request $request, Event $event)
    {
        $this->authorizeEventAccess($request, $event, 'update');

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:event_media,id'],
        ]);

        $mediaItems = EventMedia::query()
            ->where('event_id', $event->id)
            ->whereIn('id', $validated['ids'])
            ->get();

        foreach ($mediaItems as $media) {
            $media->delete();
        }

        return $this->success(['message' => 'Selected media deleted.']);
    }

    public function setCover(Request $request, Event $event, EventMedia $media)
    {
        $this->authorizeEventAccess($request, $event, 'update');

        if ($media->event_id !== $event->id || $media->file_type !== 'image') {
            return $this->error('Cover image not valid.', 422);
        }

        $event->media()->where('is_cover', true)->update(['is_cover' => false]);
        $media->update(['is_cover' => true]);

        return $this->success(['message' => 'Cover image updated.']);
    }

    public function toggleFeatured(Request $request, Event $event, EventMedia $media)
    {
        $this->authorizeEventAccess($request, $event, 'update');

        if ($media->event_id !== $event->id) {
            return $this->error('Media not found.', 404);
        }

        $media->update(['is_featured' => ! $media->is_featured]);

        return $this->success($media->fresh());
    }

    public function download(Request $request, Event $event, EventMedia $media)
    {
        $this->authorizeEventAccess($request, $event, 'view');

        if ($media->event_id !== $event->id) {
            abort(404);
        }

        $type = $request->query('type', 'optimized');

        if (! $request->user() && $type === 'original') {
            abort(403, 'Original downloads are restricted.');
        }

        if ($type === 'original') {
            return Storage::disk($media->originalDisk())->download($media->original_path, $media->file_name);
        }

        $diskName = $media->disk;
        $path = $media->optimized_path;

        if (! $path || ! Storage::disk($diskName)->exists($path)) {
            $diskName = $media->originalDisk();
            $path = $media->original_path;
        }

        return Storage::disk($diskName)->download($path, $media->file_name);
    }

    public function ingestFtp(Request $request, Event $event, EventMediaService $service)
    {
        $this->authorizeEventAccess($request, $event, 'update');

        // Enforce plan feature: FTP import (Admin/Super Admin bypass automatically)
        $planLimit = app(\App\Services\PlanLimitService::class);
        if (! $planLimit->hasFeature($request->user(), 'has_ftp_import')) {
            return $this->error('FTP tethering is not available on your current plan.', 403);
        }

        $basePath = (string) EventMediaSettings::getValue('ftp_path', config('events.media.ftp_path'));
        $basePath = rtrim($basePath, DIRECTORY_SEPARATOR);

        // FIX #21: Validate base FTP directory exists and is readable
        if (! File::isDirectory($basePath)) {
            return $this->error('FTP base directory is not configured or not found.', 422);
        }

        if (! File::isReadable($basePath)) {
            return $this->error('FTP base directory is not configured or not found.', 422);
        }

        $path = $basePath . DIRECTORY_SEPARATOR . $event->id;

        if (! File::isDirectory($path)) {
            return $this->error('FTP directory not found for this event.', 422);
        }

        $processedDir = $path . DIRECTORY_SEPARATOR . 'processed';
        File::ensureDirectoryExists($processedDir);

        $imported = 0;
        $allowed = ['jpg', 'jpeg', 'png', 'webp', 'mp4'];

        foreach (File::files($path) as $file) {
            if (! $file->isFile()) {
                continue;
            }

            $extension = strtolower($file->getExtension());
            if (! in_array($extension, $allowed, true)) {
                continue;
            }

            try {
                $media = $service->storeLocalFile($event, $file->getPathname(), $file->getFilename());
            } catch (PlanLimitReachedException $exception) {
                return $this->error($this->planLimitMessage($exception), 422);
            } catch (StorageLimitReachedException $exception) {
                return $this->error($this->storageLimitMessage($exception), 422);
            }
            if ($media) {
                $imported++;
                File::move($file->getPathname(), $processedDir . DIRECTORY_SEPARATOR . $file->getFilename());
            }
        }

        return $this->success([
            'imported' => $imported,
        ]);
    }

    private function authorizeEventAccess(Request $request, Event $event, string $ability): void
    {
        if ($request->user()) {
            $this->authorize($ability, $event);
            return;
        }

        $guestEventId = (int) $request->attributes->get('guest_event_id', 0);

        if ($guestEventId !== $event->id) {
            abort(403, 'Guest token is not valid for this event.');
        }
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'oldest' => $query->orderBy('created_at', 'asc'),
            'size_asc' => $query->orderBy('size'),
            'size_desc' => $query->orderBy('size', 'desc'),
            'name_asc' => $query->orderBy('file_name'),
            'name_desc' => $query->orderBy('file_name', 'desc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }

    private function storageLimitMessage(StorageLimitReachedException $exception): string
    {
        $usedLabel = StorageUsageService::formatBytes($exception->usedBytes());
        $limitLabel = StorageUsageService::formatBytes($exception->limitBytes());

        return 'Storage limit reached. Used ' . $usedLabel . ' of ' . $limitLabel . '.';
    }

    private function planLimitMessage(PlanLimitReachedException $exception): string
    {
        return match ($exception->limitType) {
            'storage' => 'Plan storage limit reached. Used ' . StorageUsageService::formatBytes($exception->used) . ' of ' . StorageUsageService::formatBytes($exception->limit) . '.',
            'images' => 'Plan image limit reached. You have ' . $exception->used . ' of ' . $exception->limit . ' images.',
            'events' => 'Plan event limit reached. You have ' . $exception->used . ' of ' . $exception->limit . ' active events.',
            default => $exception->getMessage(),
        };
    }
}
